/*-
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.generator.dad

import info.scce.dime.dad.dad.DAD
import info.scce.dime.data.data.UserType
import info.scce.dime.generator.process.BackendProcessGeneratorUtil
import info.scce.dime.generator.rest.DyWAAbstractGenerator
import info.scce.dime.process.process.Process
import java.nio.file.Path
import java.util.Optional

class AuthenticationGenerator extends BackendProcessGeneratorUtil {
	
	def generate(DAD dad, Path targetDir) {
		val systemUser = dad.systemUsers.stream.findFirst
		if (systemUser.isPresent) {
			val package = "info.scce.dime.auth."
			val fileName = "Authenticator.java"
			val fetchProcess = dad.findLoginUserComponents.stream.map[model].findFirst
			val content = generate(systemUser.get.systemUser, fetchProcess)
			DyWAAbstractGenerator.generate(content.toString, package, fileName, targetDir, 0)
		}
	}
	
	private def generate(UserType userType, Optional<Process> fetchProcess) '''
		// generated by «class.name»
		
		package info.scce.dime.auth;
		import info.scce.dime.config.SystemProperty;
		import info.scce.dime.config.SystemPropertyInterface;
		
		
		@javax.ws.rs.Path("/auth")
		@javax.transaction.Transactional
		public class Authenticator {
			
			private final SystemPropertyInterface serverTier = new SystemProperty("info.scce.dime.app.server.tier");
			
			«IF fetchProcess.isPresent»
				@javax.inject.Inject
				private «fetchProcess.get.typeName» process;
			«ELSE»
				@javax.inject.Inject
				private «userType.controllerTypeName» controller;
			«ENDIF»
			
			@javax.ws.rs.Path("/authenticate")
			@javax.ws.rs.POST
			@javax.ws.rs.Consumes(javax.ws.rs.core.MediaType.APPLICATION_JSON)
			public javax.ws.rs.core.Response authenticate(JWTLoginData token) throws org.apache.shiro.authc.AuthenticationException {
				
				final «userType.dyWATypeName» user;
				
				«IF fetchProcess.isPresent»
				final «fetchProcess.get.getExternalResultTypeName» result = process.execute(false, token.getUsername());
				
				switch (result.getBranchName()) {
					case "Success":
						user = result.getSuccessReturn().getUser();
						break;
					case "Not Found":
						return javax.ws.rs.core.Response.status(javax.ws.rs.core.Response.Status.UNAUTHORIZED).build();
					default:
						throw new org.apache.shiro.authc.AuthenticationException("Unexpected fetch result " + result.getBranchName());
				}
				«ELSE»
					final «userType.dyWATypeName» searchObject = controller.createSearchObject(null);
					searchObject.setusername(token.getUsername());
					
					final java.util.List<«userType.dyWATypeName»> users = controller.findByProperties(searchObject);
					
					if (users.size() > 1) {
						throw new org.apache.shiro.authc.AuthenticationException("Identification not unique");
					}
					if (users.size() == 0) {
						return javax.ws.rs.core.Response.status(javax.ws.rs.core.Response.Status.UNAUTHORIZED).build();
					}
					user = users.get(0);
				«ENDIF»
				
				java.lang.String[] passwordAndSalt = user.getpassword().split(":");
				java.lang.String password = passwordAndSalt[0];
				java.lang.String salt = passwordAndSalt[1];
				
				final org.apache.shiro.authc.SimpleAuthenticationInfo sai = new org.apache.shiro.authc.SimpleAuthenticationInfo();
				sai.setPrincipals(new org.apache.shiro.subject.SimplePrincipalCollection(user.getDywaId(), DIMERealm.REALM));
				sai.setCredentials(password);
				sai.setCredentialsSalt(org.apache.shiro.util.ByteSource.Util.bytes(java.util.Base64.getDecoder().decode(salt)));
				
				final org.apache.shiro.authc.credential.HashedCredentialsMatcher matcher = new org.apache.shiro.authc.credential.HashedCredentialsMatcher();
				matcher.setHashAlgorithmName(info.scce.dime.util.Constants.AUTH_HASH_ALGORITHM);
				matcher.setHashIterations(info.scce.dime.util.Constants.AUTH_HASH_ITERATIONS);
				matcher.setStoredCredentialsHexEncoded(info.scce.dime.util.Constants.AUTH_HEX_ENCODED);
				
				if (matcher.doCredentialsMatch(token, sai)) {
					try {
						final com.nimbusds.jose.JWSObject jwsObject = JWTUtil.createToken(user.getDywaId(), user.getfirstName(), user.getlastName());
						final java.lang.String signedToken = JWTUtil.signToken(jwsObject);
						
						return javax.ws.rs.core.Response
								.ok()
								.cookie(
									JWTCookie
										.loggedIn(
											signedToken,
											this.serverTier
										)
								 )
								.cookie(CSRFCookie.create())
								.build();
					} catch (com.nimbusds.jose.JOSEException e) {
						e.printStackTrace();
						return javax.ws.rs.core.Response.status(javax.ws.rs.core.Response.Status.UNAUTHORIZED).build();
					}
				}
				
				return javax.ws.rs.core.Response.status(javax.ws.rs.core.Response.Status.UNAUTHORIZED).build();
				
			}
			
			
			@javax.ws.rs.Path("/logout")
			@javax.ws.rs.POST
			public javax.ws.rs.core.Response logout() {
				return javax.ws.rs.core.Response
					.ok()
					.cookie(
						JWTCookie.loggedOut(
							this.serverTier
						)
					)
					.build();
			}
		}
	'''
}
