/*-
 * #%L
 * CINCO
 * %%
 * Copyright (C) 2021 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package de.jabc.cinco.meta.core.ge.style.generator.runtime.layout

import de.jabc.cinco.meta.core.ge.style.generator.runtime.layout.EdgeLayoutUtils.Location
import graphmodel.Edge

abstract class EdgeLayouter {
	
	static protected final int OFFSET_TO_NODE = 15;
	static protected final int OFFSET_TO_BENDPOINT = 10;
	
	protected extension val EdgeLayoutUtils = new EdgeLayoutUtils
	
	def void apply(Edge edge)
}

class Layouter_C_TOP extends EdgeLayouter {
	override apply(Edge it) {
		replaceBendpoints(#[
				sourceElement.topCenter,
				targetElement.topCenter
			]
			.alignTop(OFFSET_TO_NODE)
			.snapYToGrid
		)
	}
}

class Layouter_C_BOTTOM extends EdgeLayouter {
	override apply(Edge it) {
		replaceBendpoints(#[
				sourceElement.bottomCenter,
				targetElement.bottomCenter
			]
			.alignBottom(OFFSET_TO_NODE)
			.snapYToGrid
		)
	}
}

class Layouter_C_LEFT extends EdgeLayouter {
	override apply(Edge it) {
		replaceBendpoints(#[
				sourceElement.middleLeft,
				targetElement.middleLeft
			]
			.alignLeft(OFFSET_TO_NODE)
			.snapXToGrid
		)
	}
}

class Layouter_C_RIGHT extends EdgeLayouter {
	override apply(Edge it) {
		replaceBendpoints(#[
				sourceElement.middleRight,
				targetElement.middleRight
			]
			.alignRight(OFFSET_TO_NODE)
			.snapXToGrid
		)
	}
}

class Layouter_Z_HORIZONTAL extends EdgeLayouter {
	
	Location sourceLoc;
	Location targetLoc;
	
	override apply(Edge it) {
		sourceLoc = sourceElement.absoluteLocation
		targetLoc = targetElement.absoluteLocation
		if (sourceLoc.x <= targetLoc.x)
			applyLeftToRight
		 else applyRightToLeft
	}
	
	def applyLeftToRight(Edge it) {
		if (sourceLoc.y <= targetLoc.y)
			applyLeftToRightAndTopToBottom
		else applyLeftToRightAndBottomToTop
	}
	
	def applyLeftToRightAndTopToBottom(Edge it) {
		val source = sourceElement.bottomCenter
		val target = targetElement.topCenter
		val dY = Math.max(Math.abs(source.y-target.y)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toBottom(dY), target.toTop(dY))
	}
	
	def applyLeftToRightAndBottomToTop(Edge it) {
		val source = sourceElement.topCenter
		val target = targetElement.bottomCenter
		val dY = Math.max(Math.abs(source.y-target.y)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toTop(dY), target.toBottom(dY))
	}
	
	def applyRightToLeft(Edge it) {
		if (sourceLoc.y <= targetLoc.y)
			applyRightToLeftAndTopToBottom
		else applyRightToLeftAndBottomToTop
	}
	
	def applyRightToLeftAndTopToBottom(Edge it) {
		val source = sourceElement.bottomCenter
		val target = targetElement.topCenter
		val dY = Math.max(Math.abs(source.y-target.y)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toBottom(dY), target.toTop(dY))
	}
	
	def applyRightToLeftAndBottomToTop(Edge it) {
		val source = sourceElement.topCenter
		val target = targetElement.bottomCenter
		val dY = Math.max(Math.abs(source.y-target.y)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toTop(dY), target.toBottom(dY))
	}
}

class Layouter_Z_VERTICAL extends EdgeLayouter {
	
	Location sourceLoc;
	Location targetLoc;
	
	override apply(Edge it) {
		sourceLoc = sourceElement.absoluteLocation
		targetLoc = targetElement.absoluteLocation
		if (sourceLoc.y <= targetLoc.y)
			applyTopToBottom
		else applyBottomToTop
	}
	
	def applyTopToBottom(Edge it) {
		if (sourceLoc.x <= targetLoc.x)
			applyTopToBottomAndLeftToRight
		else applyTopToBottomAndRightToLeft
	}
	
	def applyTopToBottomAndLeftToRight(Edge it) {
		val source = sourceElement.middleRight
		val target = targetElement.middleLeft
		val dX = Math.max(Math.abs(source.x-target.x)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toRight(dX), target.toLeft(dX))
	}
	
	def applyTopToBottomAndRightToLeft(Edge it) {
		val source = sourceElement.middleLeft
		val target = targetElement.middleRight
		val dX = Math.max(Math.abs(source.x-target.x)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toLeft(dX), target.toRight(dX))
	}
	
	def applyBottomToTop(Edge it) {
		if (sourceLoc.x <= targetLoc.x)
			applyBottomToTopAndLeftToRight
		else applyBottomToTopAndRightToLeft
	}
	
	def applyBottomToTopAndLeftToRight(Edge it) {
		val source = sourceElement.middleRight
		val target = targetElement.middleLeft
		val dX = Math.max(Math.abs(source.x-target.x)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toRight(dX), target.toLeft(dX))
	}
	
	def applyBottomToTopAndRightToLeft(Edge it) {
		val source = sourceElement.middleLeft
		val target = targetElement.middleRight
		val dX = Math.max(Math.abs(source.x-target.x)/2, OFFSET_TO_NODE)
		replaceBendpoints(source.toLeft(dX), target.toRight(dX))
	}
}

class Layouter_TO_LEFT extends EdgeLayouter {
	override apply(Edge it) {
		moveBendpoints[it => [x = x.snapToGrid(-OFFSET_TO_BENDPOINT)]]
	}
}

class Layouter_TO_TOP extends EdgeLayouter {
	override apply(Edge it) {
		moveBendpoints[it => [y = y.snapToGrid(-OFFSET_TO_BENDPOINT)]]
	}
}

class Layouter_TO_RIGHT extends EdgeLayouter {
	override apply(Edge it) {
		moveBendpoints[it => [x = x.snapToGrid(OFFSET_TO_BENDPOINT)]]
	}
}

class Layouter_TO_BOTTOM extends EdgeLayouter {
	override apply(Edge it) {
		moveBendpoints[it => [y = y.snapToGrid(OFFSET_TO_BENDPOINT)]]
	}
}
