/*-
 * #%L
 * CINCO
 * %%
 * Copyright (C) 2021 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package de.jabc.cinco.meta.core.ge.style.generator.runtime.editor

import de.jabc.cinco.meta.runtime.xapi.CodingExtension
import de.jabc.cinco.meta.runtime.xapi.ResourceExtension
import graphmodel.Edge
import graphmodel.GraphModel
import graphmodel.Node
import graphmodel.internal.InternalEdge
import graphmodel.internal.InternalIdentifiableElement
import graphmodel.internal.InternalModelElement
import graphmodel.internal.InternalModelElementContainer
import graphmodel.internal.InternalNode
import graphmodel.internal._Point
import java.util.ArrayList
import java.util.HashMap
import java.util.Map
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.graphiti.dt.IDiagramTypeProvider
import org.eclipse.graphiti.features.IFeatureProvider
import org.eclipse.graphiti.features.context.impl.AddBendpointContext
import org.eclipse.graphiti.features.context.impl.AddConnectionContext
import org.eclipse.graphiti.features.context.impl.AddContext
import org.eclipse.graphiti.features.context.impl.AreaContext
import org.eclipse.graphiti.features.context.impl.UpdateContext
import org.eclipse.graphiti.mm.pictograms.Connection
import org.eclipse.graphiti.mm.pictograms.ContainerShape
import org.eclipse.graphiti.mm.pictograms.FreeFormConnection
import org.eclipse.graphiti.mm.pictograms.PictogramElement
import org.eclipse.graphiti.mm.pictograms.Shape
import org.eclipse.swt.SWTException

import static org.eclipse.graphiti.ui.services.GraphitiUi.getExtensionManager
import de.jabc.cinco.meta.core.ge.style.generator.runtime.features.CincoAddFeature
import de.jabc.cinco.meta.core.ge.style.generator.runtime.api.CNode

class DiagramBuilder {
	
	extension val CodingExtension = new CodingExtension
	extension val ResourceExtension = new ResourceExtension
	
	Map<InternalIdentifiableElement, PictogramElement> pes = new HashMap

	GraphModel model
	LazyDiagram diagram
	IDiagramTypeProvider dtp
	IFeatureProvider fp
	
	new(LazyDiagram diagram, GraphModel model) {
		this.diagram = diagram
		this.model = model
	}
	
	def build(Resource resource) {
		diagram.initialization = [|
			diagram.linkTo(model.getInternalElement_())
			nodes.map[getInternalElement_()].forEach[add]
			edges.map[getInternalElement_()].forEach[add]
			diagram.update
		]
		diagram.avoidInitialization = true
		resource.transact[
			resource.contents.add(0, diagram)
		]
		diagram.avoidInitialization = false
		return diagram
	}
	
	def add(InternalModelElement it) {
		switch it {
			InternalEdge: add(getAddContext)
			InternalNode: add(getAddContext(diagram))
		}
	}
	
	def void add(InternalModelElement bo, AddContext ctx) {
		if (ctx !== null) [
			bo.cache(ctx.addIfPossible)
			bo.postprocess
		].onException[switch it {
			SWTException case "Invalid thread access".equals(message): {
				bo.cache(featureProvider.getPictogramElementForBusinessObject(bo))
				bo.postprocess
			}
			default: printStackTrace
		}]
	}
	
	def postprocess(InternalModelElement bo) {
		val pe = bo.pe
		if (pe === null) {
			warn("Pictogram null. Failed to add " + bo)
		} else switch bo {
			InternalEdge: {
				addBendpoints(bo, pe)
				updateDecorators(bo, pe)
			}
			InternalModelElementContainer :
				bo.modelElements.filter(InternalNode).forEach[
					add(getAddContext(pe as ContainerShape))
				]
		}
	}
	
	
	def addBendpoints(InternalEdge edge, PictogramElement pe) {
		new ArrayList(edge.bendpoints).forEach[ point, i|
			add(point, (pe as FreeFormConnection), i)
		]
	}
	
	def add(_Point p, FreeFormConnection connection, int index) {
		val ctx = new AddBendpointContext(connection, p.x, p.y, index)
		diagramTypeProvider.diagramBehavior.executeFeature(
			featureProvider.getAddBendpointFeature(ctx), ctx);
	}
	
	def updateDecorators(InternalEdge edge, PictogramElement pe) {
		edge.decorators.forEach[ dec, i|
			updateDecorator((pe as FreeFormConnection), i, dec.locationShift?.x -> dec.locationShift?.y)
		]
	}
	
	def updateDecorator(Connection con, int index, Pair<Integer,Integer> location) {
		val ga = [	
			con.connectionDecorators?.get(index)?.graphicsAlgorithm
		].printException
		if (ga !== null && location !== null) {
			ga.x = location.key
			ga.y = location.value
		} else warn("Failed to retrieve decorator shape (index=" + index + ") of " + con)
	}
	
	def addIfPossible(AddContext ctx) {
		val ftr = featureProvider.getAddFeature(ctx) as CincoAddFeature
		if (ftr !== null) {
			ftr.setHook(false)
			if (ftr?.canAdd(ctx))
				featureProvider.diagramTypeProvider.diagramBehavior
					.executeFeature(ftr, ctx) as PictogramElement
		} else warn("Failed to retrieve AddFeature for " + ctx)
	}
	
	def getAddContext(InternalNode bo, ContainerShape target) {
		new AddContext(new AreaContext, bo) => [
			targetContainer = target
			x = bo.x
			y = bo.y
			width = bo.width
			height = bo.height
		]
	}
	
	def getAddContext(InternalEdge edge) {
		val srcAnchor = [
			(edge.sourceElement as CNode).anchor
		].onException[
			warn("Failed to retrieve source of edge: " + edge.eClass.name + " " + edge.id)
		]
		val tgtAnchor = [
			(edge.targetElement as CNode).anchor
		].onException[
			it.printStackTrace
			warn("Failed to retrieve target of edge: " + edge.eClass.name + " " + edge.id)
		]
		if (srcAnchor !== null && tgtAnchor !== null)
			new AddConnectionContext(srcAnchor, tgtAnchor) => [
				newObject = edge
			]
	}
	
	def getPe(InternalIdentifiableElement elm) {
		pes.get(elm)
	}
	
	def getNodes() {
		model.modelElements.filter(Node) //.sortBy[index]
	}
	
	def getEdges() {
		model.modelElements.filter(Edge)
	}
	
	def linkTo(PictogramElement pe, EObject bo) {
		featureProvider.link(pe,bo)
	}
	
	def getFeatureProvider() {
		fp ?: (fp = diagramTypeProvider.featureProvider)
	}
	
	def getDiagramTypeProvider() {
		dtp ?: (dtp = extensionManager.createDiagramTypeProvider(diagram, diagram.diagramTypeProviderId))
	}
	
	def cache(InternalIdentifiableElement bo, PictogramElement pe) {
		pes.put(bo, pe)
	}
	
	def clearCache() {
		pes.clear
	}
	
	def update(LazyDiagram diagram) {
		val ctx = new UpdateContext(diagram)
		val feature = featureProvider.getUpdateFeature(ctx)
		if (feature.canUpdate(ctx)) 
			feature.update(ctx)
		
		model.updateModelElements
	}
	
	def warn(String msg) {
		System.err.println("[" + class.simpleName + "] " + msg)
	}
}
