/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.util;

import com.google.common.base.Objects;
import graphmodel.IdentifiableElement;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Represents the ID of a data object to be used in the context
 * of the native domain generator. The representation there is
 * max 16 Bytes long and does not contain any special character
 * other than the underscore '_'
 */
@SuppressWarnings("all")
public class DataID {
  @Accessors
  private final String cincoId;
  
  private String _escapedLowerCase;
  
  private String _escapedLowerCase16;
  
  public static DataID from(final IdentifiableElement element) {
    String _id = element.getId();
    return new DataID(_id);
  }
  
  public static DataID from(final String cincoId) {
    return new DataID(cincoId);
  }
  
  public DataID(final String cincoId) {
    this.cincoId = cincoId;
  }
  
  /**
   * Replaces any character other than A-Z or a-z or digits 0-9 with
   * an underscore '_' and converts the ID to lower case.
   */
  private String escapedLowerCase() {
    String _elvis = null;
    if (this._escapedLowerCase != null) {
      _elvis = this._escapedLowerCase;
    } else {
      _elvis = this._escapedLowerCase = this.cincoId.replaceAll("[^\\dA-Za-z0-9]", "_").toLowerCase();
    }
    return _elvis;
  }
  
  /**
   * Escapes the given String and cuts its length if it exceeds 16 chars.
   * The escaped String contains only ASCII characters. Hence String.length() is
   * equal to String.getBytes().length() and we are safe to cut at 16 chars to
   * achieve 16 Bytes.
   */
  public String escapedLowerCase16() {
    String _elvis = null;
    if (this._escapedLowerCase16 != null) {
      _elvis = this._escapedLowerCase16;
    } else {
      String _xblockexpression = null;
      {
        final String str = this.escapedLowerCase();
        _xblockexpression = this._escapedLowerCase16 = str.substring(0, Math.min(str.length(), 16));
      }
      _elvis = _xblockexpression;
    }
    return _elvis;
  }
  
  /**
   * Compares this ID to the given one after converting both via escapedLowerCase16()
   */
  public boolean equalsEscapedLowerCase16(final DataID other) {
    String _escapedLowerCase16 = this.escapedLowerCase16();
    String _escapedLowerCase16_1 = null;
    if (other!=null) {
      _escapedLowerCase16_1=other.escapedLowerCase16();
    }
    return Objects.equal(_escapedLowerCase16, _escapedLowerCase16_1);
  }
  
  /**
   * Compares this ID to the given one after converting both via escapedLowerCase16()
   */
  public boolean equalsEscapedLowerCase16(final String otherId) {
    return this.equalsEscapedLowerCase16(DataID.from(otherId));
  }
  
  /**
   * Checks whether the ID of the given element matches any of the IDs of the given
   * elements after converting them via escapedLowerCase16(). If so, this method tries
   * to generate a new ID for the given element by EcoreUtil.generateUUID() that does
   * not match any of the IDs of the given elements after converting them via
   * escapedLowerCase16().
   */
  public static void assertUniqueDataID(final IdentifiableElement element, final Iterable<? extends IdentifiableElement> elements) {
    int i = 0;
    while (((i++ <= 1000) && IterableExtensions.exists(elements, 
      new Function1<IdentifiableElement, Boolean>() {
        @Override
        public Boolean apply(final IdentifiableElement it) {
          return Boolean.valueOf(DataID.from(it).equalsEscapedLowerCase16(element.getId()));
        }
      }))) {
      EcoreUtil.setID(element, EcoreUtil.generateUUID());
    }
    if ((i >= 1000)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("ERROR Failed to set unique DataID of ");
      String _name = element.eClass().getName();
      _builder.append(_name);
      _builder.append(" ");
      String _id = element.getId();
      _builder.append(_id);
      System.err.println(_builder);
    }
  }
  
  @Pure
  public String getCincoId() {
    return this.cincoId;
  }
}
