/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.profile.checks;

import com.google.common.collect.Iterables;
import de.jabc.cinco.meta.plugin.mcam.runtime.core.CincoCheckModule;
import graphmodel.Container;
import graphmodel.ModelElementContainer;
import graphmodel.Node;
import info.scce.dime.profile.build.PrimeSIBBuild;
import info.scce.dime.profile.checks.AbstractProfileCheck;
import info.scce.dime.profile.profile.BlueprintSIB;
import info.scce.dime.profile.profile.Branch;
import info.scce.dime.profile.profile.DoNotReplace;
import info.scce.dime.profile.profile.InputPort;
import info.scce.dime.profile.profile.OutputPort;
import info.scce.dime.profile.profile.ReplacementBranch;
import info.scce.dime.profile.profile.ReplacementCondition;
import info.scce.dime.profile.profile.ReplacementSIB;
import info.scce.dime.profile.profile.SIB;
import info.scce.dime.profile.util.ReplacementStrategy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class ReplacementCheck extends AbstractProfileCheck {
  @Extension
  private ReplacementStrategy _replacementStrategy = new ReplacementStrategy();
  
  protected boolean _checkIntegrity(final ReplacementSIB sib) {
    ModelElementContainer _referencedObject = this._profileExtension.getReferencedObject(sib);
    boolean _tripleNotEquals = (_referencedObject != null);
    if (_tripleNotEquals) {
      PrimeSIBBuild<? extends SIB, InputPort, ? extends Container, ? extends Branch, ? extends EObject, ? extends OutputPort, ? extends EObject> _compound = this._profileExtension.getCompound(sib);
      ArrayList<String> _validate = null;
      if (_compound!=null) {
        _validate=_compound.validate();
      }
      final ArrayList<String> errors = _validate;
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(errors);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("SIB is out of sync with its referenced element, please apply an update");
        this.addError(sib, _builder.toString());
        return false;
      }
    }
    final Function1<Node, Boolean> _function = new Function1<Node, Boolean>() {
      @Override
      public Boolean apply(final Node it) {
        return Boolean.valueOf(ReplacementCheck.this.checkIntegrity(it));
      }
    };
    final Function1<Boolean, Boolean> _function_1 = new Function1<Boolean, Boolean>() {
      @Override
      public Boolean apply(final Boolean it) {
        return Boolean.valueOf((it == Boolean.valueOf(true)));
      }
    };
    IterableExtensions.<Boolean>forall(IterableExtensions.<Node, Boolean>map(Iterables.<Node>filter(this._graphModelExtension.find(sib, InputPort.class, Branch.class), Node.class), _function), _function_1);
    return true;
  }
  
  protected boolean _checkIntegrity(final BlueprintSIB sib) {
    if ((this._profileExtension.hasInvalidPrimeReference(sib) && IterableExtensions.<SIB>exists(Iterables.<SIB>filter(this._graphModelExtension.findSuccessorsVia(sib, ReplacementCondition.class), SIB.class), new Function1<SIB, Boolean>() {
      @Override
      public Boolean apply(final SIB it) {
        return Boolean.valueOf(ReplacementCheck.this._replacementStrategy.isInferredReplacement(it));
      }
    }))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Blueprint SIB \'");
      String _cachedReferencedBlueprintSIBLabel = sib.getCachedReferencedBlueprintSIBLabel();
      _builder.append(_cachedReferencedBlueprintSIBLabel);
      _builder.append("\' does not exist in model \'");
      String _cachedReferencedBlueprintSIBsModelName = sib.getCachedReferencedBlueprintSIBsModelName();
      _builder.append(_cachedReferencedBlueprintSIBsModelName);
      _builder.append("\' ");
      this.addError(sib, _builder.toString());
      return false;
    }
    final Function1<Node, Boolean> _function = new Function1<Node, Boolean>() {
      @Override
      public Boolean apply(final Node it) {
        return Boolean.valueOf(ReplacementCheck.this.checkIntegrity(it));
      }
    };
    final Function1<Boolean, Boolean> _function_1 = new Function1<Boolean, Boolean>() {
      @Override
      public Boolean apply(final Boolean it) {
        return Boolean.valueOf((it == Boolean.valueOf(true)));
      }
    };
    return IterableExtensions.<Boolean>forall(IterableExtensions.<Node, Boolean>map(Iterables.<Node>filter(this._graphModelExtension.find(sib, InputPort.class, Branch.class), Node.class), _function), _function_1);
  }
  
  protected boolean _checkReplacement(final BlueprintSIB sib) {
    Node _replacement = this._replacementStrategy.getReplacement(sib);
    boolean _tripleEquals = (_replacement == null);
    if (_tripleEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("No replacement defined for SIB \'");
      String _name = this._profileExtension.getName(sib);
      _builder.append(_name);
      _builder.append("\' ");
      this.addError(sib, _builder.toString());
      return false;
    }
    ArrayList<Node> _replacementPath = this._replacementStrategy.getReplacementPath(sib);
    for (final Node replSib : _replacementPath) {
      {
        final Set<Node> cycle = this._replacementStrategy.findCycleInReplacementConditions(replSib);
        boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(cycle);
        boolean _not = (!_isNullOrEmpty);
        if (_not) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("Cycle in replacement conditions: ");
          final Function1<SIB, String> _function = new Function1<SIB, String>() {
            @Override
            public String apply(final SIB it) {
              return ReplacementCheck.this._profileExtension.getName(it);
            }
          };
          String _join = IterableExtensions.join(IterableExtensions.<SIB, String>map(Iterables.<SIB>filter(cycle, SIB.class), _function), " -> ");
          _builder_1.append(_join);
          this.addError(replSib, _builder_1.toString());
          return false;
        }
      }
    }
    final Node inferredReplacement = this._replacementStrategy.getInferredReplacement(sib);
    if ((inferredReplacement == null)) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("No replacement available for BlueprintSIB \'");
      String _name_1 = this._profileExtension.getName(sib);
      _builder_1.append(_name_1);
      _builder_1.append("\' ");
      this.addError(sib, _builder_1.toString());
      return false;
    }
    final Procedure1<BlueprintSIB> _function = new Procedure1<BlueprintSIB>() {
      @Override
      public void apply(final BlueprintSIB it) {
        boolean _replacementConditionsFulfilled = ReplacementCheck.this._replacementStrategy.getReplacementConditionsFulfilled(it);
        boolean _not = (!_replacementConditionsFulfilled);
        if (_not) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Blueprint SIB \'");
          String _name = ReplacementCheck.this._profileExtension.getName(sib);
          _builder.append(_name);
          _builder.append("\' is not going to be replaced (conditions not fulfilled). ");
          ReplacementCheck.this.addInfo(it, _builder.toString());
        } else {
          if ((inferredReplacement instanceof ReplacementSIB)) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("Inferred replacement for BlueprintSIB \'");
            String _name_1 = ReplacementCheck.this._profileExtension.getName(sib);
            _builder_1.append(_name_1);
            _builder_1.append("\' is ");
            String _name_2 = ((ReplacementSIB)inferredReplacement).eClass().getName();
            _builder_1.append(_name_2);
            _builder_1.append(" \'");
            String _name_3 = ReplacementCheck.this._profileExtension.getName(((SIB)inferredReplacement));
            _builder_1.append(_name_3);
            _builder_1.append("\' ");
            ReplacementCheck.this.addInfo(it, _builder_1.toString());
          } else {
            if ((inferredReplacement instanceof DoNotReplace)) {
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append("Blueprint SIB \'");
              String _name_4 = ReplacementCheck.this._profileExtension.getName(sib);
              _builder_2.append(_name_4);
              _builder_2.append("\' is not going to be replaced. ");
              ReplacementCheck.this.addInfo(it, _builder_2.toString());
            }
          }
        }
      }
    };
    ObjectExtensions.<BlueprintSIB>operator_doubleArrow(sib, _function);
    final Function1<Node, Boolean> _function_1 = new Function1<Node, Boolean>() {
      @Override
      public Boolean apply(final Node it) {
        return Boolean.valueOf(ReplacementCheck.this.checkReplacement(it));
      }
    };
    final Function1<Boolean, Boolean> _function_2 = new Function1<Boolean, Boolean>() {
      @Override
      public Boolean apply(final Boolean it) {
        return Boolean.valueOf((it == Boolean.valueOf(true)));
      }
    };
    return IterableExtensions.<Boolean>forall(IterableExtensions.<Node, Boolean>map(Iterables.<Node>filter(this._graphModelExtension.find(sib, InputPort.class, Branch.class), Node.class), _function_1), _function_2);
  }
  
  protected boolean _checkIntegrity(final InputPort port) {
    InputOutput.<String>println(("Check Integrity of InputPort : " + port));
    if ((this._profileExtension.hasInvalidPrimeReference(port) && (!this._profileExtension.hasInvalidPrimeReference(port.getContainer())))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Input port \'");
      String _cachedReferencedObjectName = port.getCachedReferencedObjectName();
      _builder.append(_cachedReferencedObjectName);
      _builder.append("\' does not exist. SIB out of date? ");
      this.addError(port, _builder.toString());
      return false;
    }
    return true;
  }
  
  protected boolean _checkReplacement(final InputPort port) {
    final InputPort replacement = this._replacementStrategy.getReplacement(port);
    if ((replacement == null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Replacement for input port \'");
      String _name = this._profileExtension.getName(port);
      _builder.append(_name);
      _builder.append("\' of SIB \'");
      String _name_1 = this._profileExtension.getName(this._profileExtension.getSib(port));
      _builder.append(_name_1);
      _builder.append("\' cannot be inferred");
      this.addError(port, _builder.toString());
      return false;
    }
    Object _dataType = this._profileExtension.getDataType(port);
    boolean _tripleEquals = (_dataType == null);
    if (_tripleEquals) {
      return false;
    }
    List<InputPort> _targetedPorts = this._profileExtension.getTargetedPorts(port);
    final Function1<InputPort, Boolean> _function = new Function1<InputPort, Boolean>() {
      @Override
      public Boolean apply(final InputPort it) {
        Object _dataType = ReplacementCheck.this._profileExtension.getDataType(it);
        return Boolean.valueOf((_dataType != null));
      }
    };
    final Consumer<InputPort> _function_1 = new Consumer<InputPort>() {
      @Override
      public void accept(final InputPort target) {
        final Function1<InputPort, Boolean> _function = new Function1<InputPort, Boolean>() {
          @Override
          public Boolean apply(final InputPort it) {
            return Boolean.valueOf(ReplacementCheck.this._profileExtension.isTypeOf(port, target));
          }
        };
        CincoCheckModule.Check<InputPort> _check = ReplacementCheck.this.<InputPort>check(port, _function);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Type of input port \'");
        String _name = ReplacementCheck.this._profileExtension.getName(port);
        _builder.append(_name);
        _builder.append("\' of SIB \'");
        String _name_1 = ReplacementCheck.this._profileExtension.getName(ReplacementCheck.this._profileExtension.getSib(port));
        _builder.append(_name_1);
        _builder.append("\' is incompatible to the type of the target port \'");
        String _name_2 = null;
        if (target!=null) {
          _name_2=ReplacementCheck.this._profileExtension.getName(target);
        }
        _builder.append(_name_2);
        _builder.append("\' of SIB \'");
        SIB _sib = null;
        if (target!=null) {
          _sib=ReplacementCheck.this._profileExtension.getSib(target);
        }
        String _name_3 = null;
        if (_sib!=null) {
          _name_3=ReplacementCheck.this._profileExtension.getName(_sib);
        }
        _builder.append(_name_3);
        _builder.append("\' ");
        _check.elseError(_builder.toString());
      }
    };
    IterableExtensions.<InputPort>filter(IterableExtensions.<InputPort>toSet(Iterables.<InputPort>concat(_targetedPorts, Collections.<InputPort>unmodifiableList(CollectionLiterals.<InputPort>newArrayList(replacement)))), _function).forEach(_function_1);
    return true;
  }
  
  protected boolean _checkIntegrity(final Branch branch) {
    String _name = this._profileExtension.getName(this._profileExtension.getSib(branch));
    String _plus = ("Branch " + _name);
    String _plus_1 = (_plus + ".");
    String _name_1 = this._profileExtension.getName(branch);
    String _plus_2 = (_plus_1 + _name_1);
    String _plus_3 = (_plus_2 + " hasInvalidPrimeReference: ");
    boolean _hasInvalidPrimeReference = this._profileExtension.hasInvalidPrimeReference(branch);
    String _plus_4 = (_plus_3 + Boolean.valueOf(_hasInvalidPrimeReference));
    InputOutput.<String>println(_plus_4);
    if ((this._profileExtension.hasInvalidPrimeReference(branch) && (!this._profileExtension.hasInvalidPrimeReference(branch.getContainer())))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Branch \'");
      String _cachedReferencedObjectName = branch.getCachedReferencedObjectName();
      _builder.append(_cachedReferencedObjectName);
      _builder.append("\' does not exist. SIB out of date? ");
      this.addError(branch, _builder.toString());
      return false;
    }
    final Function1<OutputPort, Boolean> _function = new Function1<OutputPort, Boolean>() {
      @Override
      public Boolean apply(final OutputPort it) {
        return Boolean.valueOf(ReplacementCheck.this.checkIntegrity(it));
      }
    };
    final Function1<Boolean, Boolean> _function_1 = new Function1<Boolean, Boolean>() {
      @Override
      public Boolean apply(final Boolean it) {
        return Boolean.valueOf((it == Boolean.valueOf(true)));
      }
    };
    return IterableExtensions.<Boolean>forall(IterableExtensions.<OutputPort, Boolean>map(this._graphModelExtension.<OutputPort>find(branch, OutputPort.class), _function), _function_1);
  }
  
  protected boolean _checkReplacement(final Branch branch) {
    String _name = this._profileExtension.getName(this._profileExtension.getSib(branch));
    String _plus = ("Branch " + _name);
    String _plus_1 = (_plus + ".");
    String _name_1 = this._profileExtension.getName(branch);
    String _plus_2 = (_plus_1 + _name_1);
    String _plus_3 = (_plus_2 + " replacement: ");
    ReplacementBranch _replacement = this._replacementStrategy.getReplacement(branch);
    String _plus_4 = (_plus_3 + _replacement);
    InputOutput.<String>println(_plus_4);
    ReplacementBranch _replacement_1 = this._replacementStrategy.getReplacement(branch);
    boolean _tripleEquals = (_replacement_1 == null);
    if (_tripleEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Replacement for branch \'");
      String _name_2 = this._profileExtension.getName(branch);
      _builder.append(_name_2);
      _builder.append("\' of SIB \'");
      String _name_3 = this._profileExtension.getName(this._profileExtension.getSib(branch));
      _builder.append(_name_3);
      _builder.append("\' cannot be inferred");
      this.addError(branch, _builder.toString());
      return false;
    }
    final Procedure1<Branch> _function = new Procedure1<Branch>() {
      @Override
      public void apply(final Branch it) {
        SIB _sib = ReplacementCheck.this._profileExtension.getSib(it);
        if ((_sib instanceof BlueprintSIB)) {
          SIB _sib_1 = ReplacementCheck.this._profileExtension.getSib(it);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Inferred replacement for branch \'");
          String _name = ReplacementCheck.this._profileExtension.getName(branch);
          _builder.append(_name);
          _builder.append("\' is branch \'");
          ReplacementBranch _inferredReplacement = ReplacementCheck.this._replacementStrategy.getInferredReplacement(branch);
          String _name_1 = null;
          if (_inferredReplacement!=null) {
            _name_1=ReplacementCheck.this._profileExtension.getName(_inferredReplacement);
          }
          _builder.append(_name_1);
          _builder.append("\' of SIB \'");
          ReplacementBranch _inferredReplacement_1 = ReplacementCheck.this._replacementStrategy.getInferredReplacement(branch);
          SIB _sib_2 = null;
          if (_inferredReplacement_1!=null) {
            _sib_2=ReplacementCheck.this._profileExtension.getSib(_inferredReplacement_1);
          }
          String _name_2 = null;
          if (_sib_2!=null) {
            _name_2=ReplacementCheck.this._profileExtension.getName(_sib_2);
          }
          _builder.append(_name_2);
          _builder.append("\' ");
          ReplacementCheck.this.addInfo(_sib_1, _builder.toString());
        }
      }
    };
    ObjectExtensions.<Branch>operator_doubleArrow(branch, _function);
    final Function1<OutputPort, Boolean> _function_1 = new Function1<OutputPort, Boolean>() {
      @Override
      public Boolean apply(final OutputPort it) {
        return Boolean.valueOf(ReplacementCheck.this.checkReplacement(it));
      }
    };
    final Function1<Boolean, Boolean> _function_2 = new Function1<Boolean, Boolean>() {
      @Override
      public Boolean apply(final Boolean it) {
        return Boolean.valueOf((it == Boolean.valueOf(true)));
      }
    };
    return IterableExtensions.<Boolean>forall(IterableExtensions.<OutputPort, Boolean>map(this._graphModelExtension.<OutputPort>find(branch, OutputPort.class), _function_1), _function_2);
  }
  
  protected boolean _checkIntegrity(final OutputPort port) {
    if ((this._profileExtension.hasInvalidPrimeReference(port) && (!this._profileExtension.hasInvalidPrimeReference(port.getContainer())))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Output port \'");
      String _cachedReferencedObjectName = port.getCachedReferencedObjectName();
      _builder.append(_cachedReferencedObjectName);
      _builder.append("\' does not exist. SIB out of date? ");
      this.addError(port, _builder.toString());
      return false;
    }
    return true;
  }
  
  protected boolean _checkReplacement(final OutputPort port) {
    final OutputPort replacement = this._replacementStrategy.getReplacement(port);
    if ((replacement == null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Replacement for output port \'");
      String _name = this._profileExtension.getName(port);
      _builder.append(_name);
      _builder.append("\' of branch \'");
      String _name_1 = this._profileExtension.getName(this._profileExtension.getBranch(port));
      _builder.append(_name_1);
      _builder.append("\' of SIB \'");
      String _name_2 = this._profileExtension.getName(this._profileExtension.getSib(this._profileExtension.getBranch(port)));
      _builder.append(_name_2);
      _builder.append("\' cannot be inferred");
      this.addError(port, _builder.toString());
      return false;
    }
    Object _dataType = this._profileExtension.getDataType(port);
    boolean _tripleEquals = (_dataType == null);
    if (_tripleEquals) {
      return false;
    }
    List<OutputPort> _targetedPorts = this._profileExtension.getTargetedPorts(port);
    final Function1<OutputPort, Boolean> _function = new Function1<OutputPort, Boolean>() {
      @Override
      public Boolean apply(final OutputPort it) {
        Object _dataType = ReplacementCheck.this._profileExtension.getDataType(it);
        return Boolean.valueOf((_dataType != null));
      }
    };
    final Consumer<OutputPort> _function_1 = new Consumer<OutputPort>() {
      @Override
      public void accept(final OutputPort target) {
        final Function1<OutputPort, Boolean> _function = new Function1<OutputPort, Boolean>() {
          @Override
          public Boolean apply(final OutputPort it) {
            return Boolean.valueOf(ReplacementCheck.this._profileExtension.isTypeOf(port, target));
          }
        };
        CincoCheckModule.Check<OutputPort> _check = ReplacementCheck.this.<OutputPort>check(port, _function);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Type of output port \'");
        String _name = ReplacementCheck.this._profileExtension.getName(port);
        _builder.append(_name);
        _builder.append("\' of branch \'");
        String _name_1 = ReplacementCheck.this._profileExtension.getName(ReplacementCheck.this._profileExtension.getBranch(port));
        _builder.append(_name_1);
        _builder.append("\' of SIB \'");
        String _name_2 = ReplacementCheck.this._profileExtension.getName(ReplacementCheck.this._profileExtension.getSib(ReplacementCheck.this._profileExtension.getBranch(port)));
        _builder.append(_name_2);
        _builder.append("\' is incompatible to the type of the target port \'");
        String _name_3 = null;
        if (target!=null) {
          _name_3=ReplacementCheck.this._profileExtension.getName(target);
        }
        _builder.append(_name_3);
        _builder.append("\' of branch \'");
        Branch _branch = null;
        if (target!=null) {
          _branch=ReplacementCheck.this._profileExtension.getBranch(target);
        }
        String _name_4 = null;
        if (_branch!=null) {
          _name_4=ReplacementCheck.this._profileExtension.getName(_branch);
        }
        _builder.append(_name_4);
        _builder.append("\' of SIB \'");
        Branch _branch_1 = null;
        if (target!=null) {
          _branch_1=ReplacementCheck.this._profileExtension.getBranch(target);
        }
        SIB _sib = null;
        if (_branch_1!=null) {
          _sib=ReplacementCheck.this._profileExtension.getSib(_branch_1);
        }
        String _name_5 = null;
        if (_sib!=null) {
          _name_5=ReplacementCheck.this._profileExtension.getName(_sib);
        }
        _builder.append(_name_5);
        _builder.append("\' ");
        _check.elseError(_builder.toString());
      }
    };
    IterableExtensions.<OutputPort>filter(IterableExtensions.<OutputPort>toSet(Iterables.<OutputPort>concat(_targetedPorts, Collections.<OutputPort>unmodifiableList(CollectionLiterals.<OutputPort>newArrayList(replacement)))), _function).forEach(_function_1);
    return true;
  }
  
  public boolean checkIntegrity(final Node sib) {
    if (sib instanceof BlueprintSIB) {
      return _checkIntegrity((BlueprintSIB)sib);
    } else if (sib instanceof InputPort) {
      return _checkIntegrity((InputPort)sib);
    } else if (sib instanceof OutputPort) {
      return _checkIntegrity((OutputPort)sib);
    } else if (sib instanceof ReplacementSIB) {
      return _checkIntegrity((ReplacementSIB)sib);
    } else if (sib instanceof Branch) {
      return _checkIntegrity((Branch)sib);
    } else if (sib != null) {
      return _checkIntegrity(sib);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(sib).toString());
    }
  }
  
  public boolean checkReplacement(final Node sib) {
    if (sib instanceof BlueprintSIB) {
      return _checkReplacement((BlueprintSIB)sib);
    } else if (sib instanceof InputPort) {
      return _checkReplacement((InputPort)sib);
    } else if (sib instanceof OutputPort) {
      return _checkReplacement((OutputPort)sib);
    } else if (sib instanceof Branch) {
      return _checkReplacement((Branch)sib);
    } else if (sib != null) {
      return _checkReplacement(sib);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(sib).toString());
    }
  }
}
