/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.process.checks;

import info.scce.dime.checks.AbstractCheck;
import info.scce.dime.data.data.Type;
import info.scce.dime.process.checks.ProcessUtils;
import info.scce.dime.process.helper.ProcessExtension;
import info.scce.dime.process.mcam.adapter.ProcessAdapter;
import info.scce.dime.process.mcam.adapter.ProcessId;
import info.scce.dime.process.process.Branch;
import info.scce.dime.process.process.ComplexInputPort;
import info.scce.dime.process.process.ComplexOutputPort;
import info.scce.dime.process.process.EndSIB;
import info.scce.dime.process.process.EntryPointProcessSIB;
import info.scce.dime.process.process.IO;
import info.scce.dime.process.process.Input;
import info.scce.dime.process.process.InputPort;
import info.scce.dime.process.process.InputStatic;
import info.scce.dime.process.process.JavaNativeInputPort;
import info.scce.dime.process.process.JavaNativeOutputPort;
import info.scce.dime.process.process.Output;
import info.scce.dime.process.process.PrimitiveInputPort;
import info.scce.dime.process.process.PrimitiveOutputPort;
import info.scce.dime.process.process.ProcessInputStatic;
import info.scce.dime.process.process.ProcessSIB;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;

@SuppressWarnings("all")
public class ProcessInterfaceCheck extends AbstractCheck<ProcessId, ProcessAdapter> {
  private ProcessUtils processUtils = new ProcessUtils();
  
  @Extension
  private ProcessExtension _processExtension = new ProcessExtension();
  
  private ProcessAdapter adapter;
  
  @Override
  public void doExecute(final ProcessAdapter adapter) {
    this.adapter = adapter;
    List<ProcessId> _entityIds = adapter.getEntityIds();
    for (final ProcessId id : _entityIds) {
      {
        Object obj = id.getElement();
        if ((obj instanceof ProcessSIB)) {
          ProcessSIB sib = ((ProcessSIB) obj);
          info.scce.dime.process.process.Process _proMod = sib.getProMod();
          boolean _tripleEquals = (_proMod == null);
          if (_tripleEquals) {
            this.addError(id, "Referenced process does not exist");
          } else {
            this.checkInputs(id, sib);
            this.checkBranches(id, sib);
            if ((obj instanceof EntryPointProcessSIB)) {
              this.checkInputs(id, ((EntryPointProcessSIB) obj));
            }
          }
        }
      }
    }
    this.processResults();
  }
  
  @Override
  public void init() {
  }
  
  private void checkInputs(final ProcessId id, final EntryPointProcessSIB sib) {
    final Predicate<InputPort> _function = new Predicate<InputPort>() {
      @Override
      public boolean test(final InputPort n) {
        return n.getIncoming().isEmpty();
      }
    };
    final Consumer<InputPort> _function_1 = new Consumer<InputPort>() {
      @Override
      public void accept(final InputPort n) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Input \'");
        String _name = n.getName();
        _builder.append(_name);
        _builder.append("\' requires incoming dataflow ");
        ProcessInterfaceCheck.this.addError(id, _builder.toString());
      }
    };
    sib.getInputPorts().stream().filter(_function).forEach(_function_1);
  }
  
  private void checkInputs(final ProcessId id, final ProcessSIB sib) {
    final Predicate<Input> _function = new Predicate<Input>() {
      @Override
      public boolean test(final Input i) {
        return (!(i instanceof ProcessInputStatic));
      }
    };
    List<Input> inputs = sib.getInputs().stream().filter(_function).collect(Collectors.<Input>toList());
    info.scce.dime.process.process.Process process = sib.getProMod();
    EList<Output> _outputs = process.getStartSIBs().get(0).getOutputs();
    for (final Output output : _outputs) {
      {
        int count = 0;
        Input oInput = null;
        for (final Input input : inputs) {
          boolean _equals = input.getName().equals(output.getName());
          if (_equals) {
            oInput = input;
            count++;
          }
        }
        if ((count > 1)) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Input \'");
          String _name = output.getName();
          _builder.append(_name);
          _builder.append("\' exists ");
          _builder.append(count);
          _builder.append(" times");
          this.addError(id, _builder.toString());
        } else {
          if ((count <= 0)) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("Input \'");
            String _name_1 = output.getName();
            _builder_1.append(_name_1);
            _builder_1.append("\' not found");
            this.addError(id, _builder_1.toString());
          } else {
            this.checkTypeOfIO(this.adapter.getIdByString(oInput.getId()), oInput, output);
            inputs.remove(oInput);
          }
        }
      }
    }
    for (final Input input : inputs) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Reference for \'");
      String _name = input.getName();
      _builder.append(_name);
      _builder.append("\' not found");
      this.addError(id, _builder.toString());
    }
  }
  
  private void checkBranches(final ProcessId sibId, final ProcessSIB sib) {
    EList<Branch> _successors = sib.<Branch>getSuccessors(Branch.class);
    ArrayList<Branch> branches = new ArrayList<Branch>(_successors);
    info.scce.dime.process.process.Process process = sib.getProMod();
    EList<EndSIB> _endSIBs = process.getEndSIBs();
    for (final EndSIB endSib : _endSIBs) {
      {
        int count = 0;
        Branch oBranch = null;
        for (final Branch branch : branches) {
          boolean _equals = branch.getName().equals(endSib.getBranchName());
          if (_equals) {
            oBranch = branch;
            count++;
          }
        }
        if ((count > 1)) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Branch \'");
          String _branchName = endSib.getBranchName();
          _builder.append(_branchName);
          _builder.append("\' exists ");
          _builder.append(count);
          _builder.append(" times");
          this.addError(sibId, _builder.toString());
        } else {
          if ((count <= 0)) {
            boolean _isIgnoredBranch = this._processExtension.isIgnoredBranch(sib, endSib.getBranchName());
            boolean _not = (!_isIgnoredBranch);
            if (_not) {
              StringConcatenation _builder_1 = new StringConcatenation();
              _builder_1.append("Branch \'");
              String _branchName_1 = endSib.getBranchName();
              _builder_1.append(_branchName_1);
              _builder_1.append("\' not found");
              this.addError(sibId, _builder_1.toString());
            }
          } else {
            this.checkOutputs(this.adapter.getIdByString(oBranch.getId()), oBranch, endSib);
            branches.remove(oBranch);
          }
        }
      }
    }
    for (final Branch branch : branches) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Reference for Branch \'");
      String _name = branch.getName();
      _builder.append(_name);
      _builder.append("\' not found");
      this.addError(sibId, _builder.toString());
    }
  }
  
  private void checkOutputs(final ProcessId branchId, final Branch branch, final EndSIB endSib) {
    EList<Output> _outputs = branch.getOutputs();
    ArrayList<Output> outputs = new ArrayList<Output>(_outputs);
    EList<Input> _inputs = endSib.getInputs();
    for (final Input input : _inputs) {
      {
        int count = 0;
        Output oOutput = null;
        for (final Output output : outputs) {
          boolean _equals = output.getName().equals(input.getName());
          if (_equals) {
            oOutput = output;
            count++;
          }
        }
        if ((count > 1)) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Output \'");
          String _name = input.getName();
          _builder.append(_name);
          _builder.append("\' exists ");
          _builder.append(count);
          _builder.append(" times");
          this.addError(branchId, _builder.toString());
        } else {
          if ((count <= 0)) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("Output \'");
            String _name_1 = input.getName();
            _builder_1.append(_name_1);
            _builder_1.append("\' not found");
            this.addError(branchId, _builder_1.toString());
          } else {
            this.checkTypeOfIO(this.adapter.getIdByString(oOutput.getId()), oOutput, input);
            outputs.remove(oOutput);
          }
        }
      }
    }
    for (final Output output : outputs) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Reference for \'");
      String _name = output.getName();
      _builder.append(_name);
      _builder.append("\' not found");
      this.addError(branchId, _builder.toString());
    }
  }
  
  private void checkTypeOfIO(final ProcessId thisId, final IO thisIO, final IO referencedIO) {
    Object thisType = null;
    Object refType = null;
    if (((thisIO instanceof PrimitiveOutputPort) || (thisIO instanceof PrimitiveInputPort))) {
      thisType = this.processUtils.getPrimitiveTypeString(thisIO);
    } else {
      if (((thisIO instanceof ComplexOutputPort) || (thisIO instanceof ComplexInputPort))) {
        thisType = this.processUtils.getComplexType(thisIO);
      } else {
        if (((thisIO instanceof JavaNativeOutputPort) || (thisIO instanceof JavaNativeInputPort))) {
          thisType = this.processUtils.getNativeType(thisIO);
        } else {
          if ((thisIO instanceof InputStatic)) {
            thisType = this.processUtils.getStaticTypeString(thisIO);
          } else {
            throw new RuntimeException("Could not find port type");
          }
        }
      }
    }
    if (((referencedIO instanceof PrimitiveOutputPort) || (referencedIO instanceof PrimitiveInputPort))) {
      refType = this.processUtils.getPrimitiveTypeString(referencedIO);
    } else {
      if (((referencedIO instanceof ComplexOutputPort) || (referencedIO instanceof ComplexInputPort))) {
        refType = this.processUtils.getComplexType(referencedIO);
      } else {
        if (((referencedIO instanceof JavaNativeOutputPort) || (referencedIO instanceof JavaNativeInputPort))) {
          refType = this.processUtils.getNativeType(referencedIO);
        } else {
          if ((referencedIO instanceof InputStatic)) {
            refType = this.processUtils.getStaticTypeString(referencedIO);
          } else {
            throw new RuntimeException("Could not find port type");
          }
        }
      }
    }
    if (((thisType instanceof String) && (refType instanceof String))) {
      boolean _equals = ((String) thisType).equals(refType);
      boolean _not = (!_equals);
      if (_not) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Type mismatch, should be \'");
        _builder.append(((String) refType));
        _builder.append("\'");
        this.addError(thisId, _builder.toString());
      }
    } else {
      if (((thisType instanceof Type) && (refType instanceof Type))) {
        boolean _equals_1 = ((Type) thisType).getId().equals(((Type) refType).getId());
        boolean _not_1 = (!_equals_1);
        if (_not_1) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("Type mismatch, should be \'");
          String _name = ((Type) refType).getName();
          _builder_1.append(_name);
          _builder_1.append("\'");
          this.addError(thisId, _builder_1.toString());
        }
      } else {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("Type mismatch, should be \'");
        String _simpleName = refType.getClass().getSimpleName();
        _builder_2.append(_simpleName);
        _builder_2.append("\'");
        this.addError(thisId, _builder_2.toString());
      }
    }
    boolean _isListType = this.processUtils.isListType(thisIO);
    boolean _isListType_1 = this.processUtils.isListType(referencedIO);
    boolean _tripleNotEquals = (Boolean.valueOf(_isListType) != Boolean.valueOf(_isListType_1));
    if (_tripleNotEquals) {
      StringConcatenation _builder_3 = new StringConcatenation();
      _builder_3.append("isList property mismatch, should be ");
      boolean _isListType_2 = this.processUtils.isListType(referencedIO);
      _builder_3.append(_isListType_2);
      this.addError(thisId, _builder_3.toString());
    }
  }
}
