/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.process.checks;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import graphmodel.Container;
import graphmodel.ModelElementContainer;
import info.scce.dime.checks.AbstractCheck;
import info.scce.dime.data.data.PrimitiveType;
import info.scce.dime.data.data.Type;
import info.scce.dime.data.helper.DataExtension;
import info.scce.dime.gUIPlugin.AbstractParameter;
import info.scce.dime.gUIPlugin.Function;
import info.scce.dime.gui.gui.AddToSubmission;
import info.scce.dime.gui.gui.Attribute;
import info.scce.dime.gui.gui.Button;
import info.scce.dime.gui.gui.ButtonOptions;
import info.scce.dime.gui.gui.ComplexAttribute;
import info.scce.dime.gui.gui.ComplexAttributeConnector;
import info.scce.dime.gui.gui.ComplexListAttribute;
import info.scce.dime.gui.gui.ComplexListAttributeConnector;
import info.scce.dime.gui.gui.ComplexOutputPort;
import info.scce.dime.gui.gui.ComplexVariable;
import info.scce.dime.gui.gui.DataContext;
import info.scce.dime.gui.gui.Event;
import info.scce.dime.gui.gui.GUI;
import info.scce.dime.gui.gui.GUIPlugin;
import info.scce.dime.gui.gui.ListenerContext;
import info.scce.dime.gui.gui.OutputPort;
import info.scce.dime.gui.gui.PrimitiveAttribute;
import info.scce.dime.gui.gui.PrimitiveListAttribute;
import info.scce.dime.gui.gui.PrimitiveOutputPort;
import info.scce.dime.gui.gui.PrimitiveVariable;
import info.scce.dime.gui.gui.Variable;
import info.scce.dime.gui.helper.ComplexGUIBranchPort;
import info.scce.dime.gui.helper.GUIBranch;
import info.scce.dime.gui.helper.GUIBranchPort;
import info.scce.dime.gui.helper.GUIExtension;
import info.scce.dime.gui.helper.GUIExtensionProvider;
import info.scce.dime.process.helper.ProcessExtension;
import info.scce.dime.process.mcam.adapter.ProcessAdapter;
import info.scce.dime.process.mcam.adapter.ProcessId;
import info.scce.dime.process.process.BooleanInputStatic;
import info.scce.dime.process.process.Branch;
import info.scce.dime.process.process.ComplexInputPort;
import info.scce.dime.process.process.EventListener;
import info.scce.dime.process.process.GUISIB;
import info.scce.dime.process.process.Input;
import info.scce.dime.process.process.InputPort;
import info.scce.dime.process.process.InputStatic;
import info.scce.dime.process.process.IntegerInputStatic;
import info.scce.dime.process.process.Output;
import info.scce.dime.process.process.PrimitiveInputPort;
import info.scce.dime.process.process.RealInputStatic;
import info.scce.dime.process.process.TextInputStatic;
import info.scce.dime.process.process.TimestampInputStatic;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.jooq.lambda.Seq;

@SuppressWarnings("all")
public class GuiInterfaceCheck extends AbstractCheck<ProcessId, ProcessAdapter> {
  @Extension
  private DataExtension _dataExtension = DataExtension.getInstance();
  
  @Extension
  private ProcessExtension _processExtension = new ProcessExtension();
  
  @Extension
  private GUIExtension guiex;
  
  private ProcessAdapter adapter;
  
  private GUISIB sib;
  
  private GUI refGui;
  
  private GUI cGui;
  
  @Override
  public void doExecute(final ProcessAdapter adapter) {
    this.adapter = adapter;
    List<ProcessId> _entityIds = adapter.getEntityIds();
    for (final ProcessId id : _entityIds) {
      {
        Object obj = id.getElement();
        if ((obj instanceof GUISIB)) {
          this.sib = ((GUISIB) obj);
          this.refGui = this.sib.getGui();
          this.cGui = this.refGui;
          this.checkInputs(id, this.sib);
          this.checkEvents(id, this.sib);
          this.checkBranches(id, this.sib);
          this.checkBranchPorts();
        }
      }
    }
    this.processResults();
  }
  
  @Override
  public void init() {
    this.guiex = GUIExtensionProvider.guiextension();
  }
  
  public void checkEvents(final ProcessId id, final GUISIB guisib) {
    final EList<EventListener> events = guisib.getEventListenerSuccessors();
    final Function1<ListenerContext, EList<Event>> _function = new Function1<ListenerContext, EList<Event>>() {
      @Override
      public EList<Event> apply(final ListenerContext n) {
        return n.getEvents();
      }
    };
    final Iterable<Event> listener = Iterables.<Event>concat(ListExtensions.<ListenerContext, EList<Event>>map(this.refGui.getListenerContexts(), _function));
    final Function1<EventListener, Boolean> _function_1 = new Function1<EventListener, Boolean>() {
      @Override
      public Boolean apply(final EventListener n) {
        final Function1<Event, Boolean> _function = new Function1<Event, Boolean>() {
          @Override
          public Boolean apply(final Event it) {
            return Boolean.valueOf(it.getName().equals(n.getName()));
          }
        };
        return Boolean.valueOf(IterableExtensions.isEmpty(IterableExtensions.<Event>filter(listener, _function)));
      }
    };
    final Consumer<EventListener> _function_2 = new Consumer<EventListener>() {
      @Override
      public void accept(final EventListener n) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("event \'");
        String _name = n.getName();
        _builder.append(_name);
        _builder.append("\' not found in GUI");
        GuiInterfaceCheck.this.addError(id, _builder.toString());
      }
    };
    IterableExtensions.<EventListener>filter(events, _function_1).forEach(_function_2);
    final Function1<EventListener, Boolean> _function_3 = new Function1<EventListener, Boolean>() {
      @Override
      public Boolean apply(final EventListener n) {
        final Function1<Event, Boolean> _function = new Function1<Event, Boolean>() {
          @Override
          public Boolean apply(final Event it) {
            return Boolean.valueOf(it.getName().equals(n.getName()));
          }
        };
        boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<Event>filter(listener, _function));
        return Boolean.valueOf((!_isEmpty));
      }
    };
    final Iterable<EventListener> presentEvents = IterableExtensions.<EventListener>filter(events, _function_3);
    for (final EventListener pEvent : presentEvents) {
      {
        final Function1<Event, Boolean> _function_4 = new Function1<Event, Boolean>() {
          @Override
          public Boolean apply(final Event it) {
            return Boolean.valueOf(it.getName().equals(pEvent.getName()));
          }
        };
        final Event coListener = IterableExtensions.<Event>findFirst(listener, _function_4);
        final Function1<OutputPort, Boolean> _function_5 = new Function1<OutputPort, Boolean>() {
          @Override
          public Boolean apply(final OutputPort it) {
            final Function1<Input, String> _function = new Function1<Input, String>() {
              @Override
              public String apply(final Input it) {
                return it.getName();
              }
            };
            boolean _contains = ListExtensions.<Input, String>map(pEvent.getInputs(), _function).contains(it.getName());
            return Boolean.valueOf((!_contains));
          }
        };
        final Consumer<OutputPort> _function_6 = new Consumer<OutputPort>() {
          @Override
          public void accept(final OutputPort n) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("missing port ");
            String _name = n.getName();
            _builder.append(_name);
            GuiInterfaceCheck.this.addError(id, _builder.toString());
          }
        };
        IterableExtensions.<OutputPort>filter(coListener.getOutputPorts(), _function_5).forEach(_function_6);
        final Function1<Input, Boolean> _function_7 = new Function1<Input, Boolean>() {
          @Override
          public Boolean apply(final Input it) {
            final Function1<OutputPort, String> _function = new Function1<OutputPort, String>() {
              @Override
              public String apply(final OutputPort it) {
                return it.getName();
              }
            };
            boolean _contains = ListExtensions.<OutputPort, String>map(coListener.getOutputPorts(), _function).contains(it.getName());
            return Boolean.valueOf((!_contains));
          }
        };
        final Consumer<Input> _function_8 = new Consumer<Input>() {
          @Override
          public void accept(final Input n) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("unkown port ");
            String _name = n.getName();
            _builder.append(_name);
            GuiInterfaceCheck.this.addError(id, _builder.toString());
          }
        };
        IterableExtensions.<Input>filter(pEvent.getInputs(), _function_7).forEach(_function_8);
        final Function1<Input, OutputPort> _function_9 = new Function1<Input, OutputPort>() {
          @Override
          public OutputPort apply(final Input n) {
            final Function1<OutputPort, Boolean> _function = new Function1<OutputPort, Boolean>() {
              @Override
              public Boolean apply(final OutputPort it) {
                return Boolean.valueOf(it.getName().equals(n.getName()));
              }
            };
            return IterableExtensions.<OutputPort>findFirst(coListener.getOutputPorts(), _function);
          }
        };
        Set<Map.Entry<OutputPort, List<Input>>> _entrySet = IterableExtensions.<OutputPort, Input>groupBy(pEvent.getInputs(), _function_9).entrySet();
        for (final Map.Entry<OutputPort, List<Input>> portTupel : _entrySet) {
          {
            final OutputPort guiListnerPort = portTupel.getKey();
            final Input guiEventPort = portTupel.getValue().get(0);
            if ((guiEventPort instanceof InputPort)) {
              boolean _isIsList = guiListnerPort.isIsList();
              boolean _isIsList_1 = ((InputPort)guiEventPort).isIsList();
              boolean _notEquals = (_isIsList != _isIsList_1);
              if (_notEquals) {
                StringConcatenation _builder = new StringConcatenation();
                _builder.append("wrong list status ");
                String _name = ((InputPort)guiEventPort).getName();
                _builder.append(_name);
                this.addError(id, _builder.toString());
              }
            }
            if ((guiListnerPort instanceof ComplexOutputPort)) {
              if ((guiEventPort instanceof ComplexInputPort)) {
                Type _dataType = ((ComplexInputPort) guiEventPort).getDataType();
                Seq<Type> _superTypes = this._dataExtension.getSuperTypes(((ComplexInputPort) guiEventPort).getDataType());
                final Iterable<Type> superTypesAndMe = Iterables.<Type>concat(Collections.<Type>unmodifiableList(CollectionLiterals.<Type>newArrayList(_dataType)), _superTypes);
                final Function1<Type, String> _function_10 = new Function1<Type, String>() {
                  @Override
                  public String apply(final Type it) {
                    return it.getName();
                  }
                };
                final Function1<String, Boolean> _function_11 = new Function1<String, Boolean>() {
                  @Override
                  public Boolean apply(final String n) {
                    return Boolean.valueOf(n.equals(((ComplexOutputPort) guiListnerPort).getDataType().getName()));
                  }
                };
                String _findFirst = IterableExtensions.<String>findFirst(IterableExtensions.<Type, String>map(superTypesAndMe, _function_10), _function_11);
                boolean _equals = Objects.equal(_findFirst, null);
                if (_equals) {
                  StringConcatenation _builder_1 = new StringConcatenation();
                  _builder_1.append("incompatible complex types ");
                  String _name_1 = ((ComplexInputPort)guiEventPort).getName();
                  _builder_1.append(_name_1);
                  this.addError(id, _builder_1.toString());
                }
              } else {
                StringConcatenation _builder_2 = new StringConcatenation();
                String _name_2 = guiEventPort.getName();
                _builder_2.append(_name_2);
                _builder_2.append(" has to be complex");
                this.addError(id, _builder_2.toString());
              }
            }
            if ((guiListnerPort instanceof PrimitiveOutputPort)) {
              if ((guiEventPort instanceof PrimitiveInputPort)) {
                PrimitiveType _data = this._dataExtension.toData(((PrimitiveOutputPort) guiListnerPort).getDataType());
                PrimitiveType _data_1 = this._dataExtension.toData(((PrimitiveInputPort) guiEventPort).getDataType());
                boolean _notEquals_1 = (!Objects.equal(_data, _data_1));
                if (_notEquals_1) {
                  StringConcatenation _builder_3 = new StringConcatenation();
                  _builder_3.append("incompatible primitive types ");
                  String _name_3 = ((PrimitiveInputPort)guiEventPort).getName();
                  _builder_3.append(_name_3);
                  this.addError(id, _builder_3.toString());
                }
              } else {
                if ((guiEventPort instanceof InputStatic)) {
                  final PrimitiveType pType = this._dataExtension.toData(((PrimitiveOutputPort)guiListnerPort).getDataType());
                  boolean _switchResult = false;
                  boolean _matched = false;
                  if (guiEventPort instanceof TextInputStatic) {
                    boolean _equals_1 = Objects.equal(pType, PrimitiveType.TEXT);
                    if (_equals_1) {
                      _matched=true;
                      _switchResult = false;
                    }
                  }
                  if (!_matched) {
                    if (guiEventPort instanceof IntegerInputStatic) {
                      boolean _equals_1 = Objects.equal(pType, PrimitiveType.INTEGER);
                      if (_equals_1) {
                        _matched=true;
                        _switchResult = false;
                      }
                    }
                  }
                  if (!_matched) {
                    if (guiEventPort instanceof RealInputStatic) {
                      boolean _equals_1 = Objects.equal(pType, PrimitiveType.REAL);
                      if (_equals_1) {
                        _matched=true;
                        _switchResult = false;
                      }
                    }
                  }
                  if (!_matched) {
                    if (guiEventPort instanceof TimestampInputStatic) {
                      boolean _equals_1 = Objects.equal(pType, PrimitiveType.TIMESTAMP);
                      if (_equals_1) {
                        _matched=true;
                        _switchResult = false;
                      }
                    }
                  }
                  if (!_matched) {
                    if (guiEventPort instanceof BooleanInputStatic) {
                      boolean _equals_1 = Objects.equal(pType, PrimitiveType.BOOLEAN);
                      if (_equals_1) {
                        _matched=true;
                        _switchResult = false;
                      }
                    }
                  }
                  final boolean isWrong = _switchResult;
                  if (isWrong) {
                    StringConcatenation _builder_4 = new StringConcatenation();
                    String _name_4 = ((InputStatic)guiEventPort).getName();
                    _builder_4.append(_name_4);
                    _builder_4.append(" primitive type incompatible");
                    this.addError(id, _builder_4.toString());
                  }
                } else {
                  StringConcatenation _builder_5 = new StringConcatenation();
                  String _name_5 = guiEventPort.getName();
                  _builder_5.append(_name_5);
                  _builder_5.append(" has to be primitive");
                  this.addError(id, _builder_5.toString());
                }
              }
            }
          }
        }
      }
    }
  }
  
  private void checkInputs(final ProcessId sibId, final GUISIB sib) {
    EList<Input> _inputs = sib.getInputs();
    List<EObject> inputs = new ArrayList<EObject>(_inputs);
    List<EObject> inputOrigins = this.collectInputs();
    for (final EObject input : inputs) {
      {
        EObject eObj = this.getRelatedPort(inputOrigins, input);
        if (((eObj == null) && (input instanceof Input))) {
          ProcessId inputId = this.adapter.getIdByString(((Input) input).getId());
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("input \'");
          String _name = ((Input) input).getName();
          _builder.append(_name);
          _builder.append("\' not found in GUI");
          this.addError(inputId, _builder.toString());
        }
      }
    }
    EList<Input> _inputs_1 = sib.getInputs();
    for (final Input input_1 : _inputs_1) {
      {
        EObject eObj = this.getRelatedPort(inputOrigins, input_1);
        boolean _notEquals = (!Objects.equal(eObj, null));
        if (_notEquals) {
          ProcessId inputId = this.adapter.getIdByString(((Input) input_1).getId());
          this.checkInput(inputId, input_1, eObj);
        }
      }
    }
    for (final EObject eObj : inputOrigins) {
      {
        EObject inputEObj = this.getRelatedPort(inputs, eObj);
        if ((inputEObj == null)) {
          String missingName = ((Variable) eObj).getName();
          this.addError(sibId, (("input \'" + missingName) + "\' is missing"));
        }
      }
    }
  }
  
  private void checkInput(final ProcessId inputId, final Input input, final EObject eObj) {
    boolean inputIsList = false;
    String inputPType = null;
    Type inputType = null;
    if ((input instanceof InputStatic)) {
      if ((input instanceof BooleanInputStatic)) {
        inputPType = info.scce.dime.process.process.PrimitiveType.BOOLEAN.getLiteral();
      } else {
        if ((input instanceof IntegerInputStatic)) {
          inputPType = info.scce.dime.process.process.PrimitiveType.INTEGER.getLiteral();
        } else {
          if ((input instanceof RealInputStatic)) {
            inputPType = info.scce.dime.process.process.PrimitiveType.REAL.getLiteral();
          } else {
            if ((input instanceof TextInputStatic)) {
              inputPType = info.scce.dime.process.process.PrimitiveType.TEXT.getLiteral();
            } else {
              if ((input instanceof TimestampInputStatic)) {
                inputPType = info.scce.dime.process.process.PrimitiveType.TIMESTAMP.getLiteral();
              } else {
                throw new IllegalStateException("else in exhaustive if should not happen; please fix broken implementation");
              }
            }
          }
        }
      }
    } else {
      if ((input instanceof PrimitiveInputPort)) {
        inputPType = ((PrimitiveInputPort) input).getDataType().getLiteral();
        inputIsList = ((PrimitiveInputPort) input).isIsList();
      } else {
        if ((input instanceof ComplexInputPort)) {
          inputType = ((ComplexInputPort) input).getDataType();
          inputIsList = ((ComplexInputPort) input).isIsList();
        } else {
          throw new IllegalStateException("else in exhaustive if should not happen; please fix broken implementation");
        }
      }
    }
    boolean originIsList = false;
    String originPType = null;
    Type originType = null;
    if ((eObj instanceof Variable)) {
      if ((eObj instanceof PrimitiveVariable)) {
        originPType = ((PrimitiveVariable) eObj).getDataType().getLiteral();
        originIsList = ((PrimitiveVariable) eObj).isIsList();
      } else {
        if ((eObj instanceof ComplexVariable)) {
          originType = ((ComplexVariable) eObj).getDataType();
          originIsList = ((ComplexVariable) eObj).isIsList();
        } else {
          throw new IllegalStateException("else in exhaustive if should not happen; please fix broken implementation");
        }
      }
    } else {
      if ((eObj instanceof Attribute)) {
        if ((eObj instanceof PrimitiveAttribute)) {
          originPType = ((PrimitiveAttribute) eObj).getAttribute().getDataType().getLiteral();
          originIsList = ((PrimitiveVariable) eObj).isIsList();
        }
        if ((eObj instanceof PrimitiveListAttribute)) {
          originPType = "integer";
          originIsList = ((PrimitiveVariable) eObj).isIsList();
        }
        if ((eObj instanceof ComplexAttribute)) {
          originType = ((ComplexAttribute) eObj).getAttribute().getDataType();
          originIsList = ((ComplexAttribute) eObj).getAttribute().isIsList();
        }
        if ((eObj instanceof ComplexListAttribute)) {
          originType = ((ComplexListAttribute) eObj).getListType();
          originIsList = false;
        }
      } else {
        if ((eObj instanceof info.scce.dime.process.process.PrimitiveOutputPort)) {
          originPType = ((info.scce.dime.process.process.PrimitiveOutputPort) eObj).getDataType().getLiteral();
          originIsList = ((info.scce.dime.process.process.PrimitiveOutputPort) eObj).isIsList();
        } else {
          if ((eObj instanceof info.scce.dime.process.process.ComplexOutputPort)) {
            originType = ((info.scce.dime.process.process.ComplexOutputPort) eObj).getDataType();
            originIsList = ((info.scce.dime.process.process.ComplexOutputPort) eObj).isIsList();
          } else {
            throw new IllegalStateException("else in exhaustive if should not happen; please fix broken implementation");
          }
        }
      }
    }
    if ((Boolean.valueOf(inputIsList) != Boolean.valueOf(originIsList))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("isList property mismatch. should be ");
      _builder.append(originIsList);
      this.addError(inputId, _builder.toString());
    }
    if (((inputPType != originPType) && (originPType != null))) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("primitive type mismatch. should be ");
      _builder_1.append(originPType);
      this.addError(inputId, _builder_1.toString());
    }
    if (((originType != null) && ((inputType == null) || (inputType.getId() != originType.getId())))) {
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("type mismatch. should be ");
      String _name = originType.getName();
      _builder_2.append(_name);
      this.addError(inputId, _builder_2.toString());
    }
  }
  
  private List<EObject> collectInputs() {
    ArrayList<EObject> list = new ArrayList<EObject>();
    EList<DataContext> _dataContexts = this.sib.getGui().getDataContexts();
    for (final DataContext dataContext : _dataContexts) {
      {
        final Predicate<Variable> _function = new Predicate<Variable>() {
          @Override
          public boolean test(final Variable var) {
            return var.isIsInput();
          }
        };
        List<Variable> inputs = dataContext.getVariables().stream().filter(_function).collect(Collectors.<Variable>toList());
        ArrayList<Variable> _arrayList = new ArrayList<Variable>(inputs);
        for (final Variable var : _arrayList) {
          {
            int _size = var.<ComplexAttributeConnector>getIncoming(ComplexAttributeConnector.class).size();
            boolean _greaterThan = (_size > 0);
            if (_greaterThan) {
              inputs.remove(var);
            }
            int _size_1 = var.<ComplexListAttributeConnector>getIncoming(ComplexListAttributeConnector.class).size();
            boolean _greaterThan_1 = (_size_1 > 0);
            if (_greaterThan_1) {
              inputs.remove(var);
            }
          }
        }
        list.addAll(inputs);
      }
    }
    return list;
  }
  
  private void checkBranches(final ProcessId sibId, final GUISIB sib) {
    List<Branch> branches = sib.<Branch>getSuccessors(Branch.class);
    final Function1<GUIBranch, String> _function = new Function1<GUIBranch, String>() {
      @Override
      public String apply(final GUIBranch it) {
        return it.getName();
      }
    };
    Set<String> branchesOrigins = IterableExtensions.<String>toSet(ListExtensions.<GUIBranch, String>map(this.guiex.getGUIBranches(sib.getGui(), true), _function));
    for (final Branch branch : branches) {
      boolean _contains = branchesOrigins.contains(branch.getName());
      boolean _not = (!_contains);
      if (_not) {
        ProcessId _idByString = this.adapter.getIdByString(branch.getId());
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("branch \'");
        String _name = branch.getName();
        _builder.append(_name);
        _builder.append("\' not found in GUI");
        this.addError(_idByString, _builder.toString());
      }
    }
    for (final String branchName : branchesOrigins) {
      if (((!IterableExtensions.<Branch>exists(branches, new Function1<Branch, Boolean>() {
        @Override
        public Boolean apply(final Branch it) {
          String _name = it.getName();
          return Boolean.valueOf(Objects.equal(_name, branchName));
        }
      })) && (!this._processExtension.isIgnoredBranch(sib, branchName)))) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("Branch \'");
        _builder_1.append(branchName);
        _builder_1.append("\' is missing");
        this.addError(sibId, _builder_1.toString());
      }
    }
  }
  
  private void checkBranchPorts() {
    EList<Branch> _successors = this.sib.<Branch>getSuccessors(Branch.class);
    for (final Branch branch : _successors) {
      {
        ProcessId branchId = this.adapter.getIdByString(branch.getId());
        final Function1<GUIBranch, Boolean> _function = new Function1<GUIBranch, Boolean>() {
          @Override
          public Boolean apply(final GUIBranch it) {
            String _name = it.getName();
            String _name_1 = branch.getName();
            return Boolean.valueOf(Objects.equal(_name, _name_1));
          }
        };
        final GUIBranch branchRepresentant = IterableExtensions.<GUIBranch>findFirst(this.guiex.getGUIBranchesMerged(this.sib.getGui()), _function);
        if ((branchRepresentant != null)) {
          this.checkOutputs(branchId, branch, branchRepresentant);
        }
      }
    }
  }
  
  private void checkOutputs(final ProcessId branchId, final Branch branch, final GUIBranch branchRep) {
    EList<Output> _outputs = branch.getOutputs();
    final List<Output> outputs = new ArrayList<Output>(_outputs);
    final Function1<GUIBranchPort, String> _function = new Function1<GUIBranchPort, String>() {
      @Override
      public String apply(final GUIBranchPort it) {
        return it.getName();
      }
    };
    List<String> _map = ListExtensions.<GUIBranchPort, String>map(branchRep.getPorts(), _function);
    final List<String> names = new LinkedList<String>(_map);
    for (final Output output : outputs) {
      boolean _contains = names.contains(output.getName());
      boolean _not = (!_contains);
      if (_not) {
        ProcessId _idByString = this.adapter.getIdByString(output.getId());
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("output \'");
        String _name = output.getName();
        _builder.append(_name);
        _builder.append("\' not found in GUI");
        this.addError(_idByString, _builder.toString());
      } else {
        final Function1<GUIBranchPort, Boolean> _function_1 = new Function1<GUIBranchPort, Boolean>() {
          @Override
          public Boolean apply(final GUIBranchPort it) {
            return Boolean.valueOf(it.getName().equals(output.getName()));
          }
        };
        final GUIBranchPort portObj = IterableExtensions.<GUIBranchPort>findFirst(branchRep.getPorts(), _function_1);
        final ProcessId outputId = this.adapter.getIdByString(output.getId());
        this.checkOutput(outputId, output, portObj);
        names.remove(output.getName());
      }
    }
    for (final String name : names) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("output \'");
      _builder_1.append(name);
      _builder_1.append("\' is missing");
      this.addError(branchId, _builder_1.toString());
    }
  }
  
  private Set<Button> findButtons(final ModelElementContainer container) {
    if ((container instanceof Button)) {
      ButtonOptions _options = ((Button) container).getOptions();
      boolean _tripleNotEquals = (_options != null);
      if (_tripleNotEquals) {
        String _staticURL = ((Button) container).getOptions().getStaticURL();
        boolean _tripleNotEquals_1 = (_staticURL != null);
        if (_tripleNotEquals_1) {
          boolean _isEmpty = ((Button) container).getOptions().getStaticURL().isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            Set<Button> empty = Collections.<Button>emptySet();
            return empty;
          }
        }
      }
      boolean _isDisabled = ((Button) container).isDisabled();
      if (_isDisabled) {
        return Collections.<Button>emptySet();
      }
      return Collections.<Button>singleton(((Button) container));
    } else {
      Set<Button> contained = new HashSet<Button>();
      EList<Container> _allContainers = container.getAllContainers();
      for (final Container c : _allContainers) {
        contained.addAll(this.findButtons(c));
      }
      return contained;
    }
  }
  
  private Set<info.scce.dime.gUIPlugin.Output> findGUIPlugins(final ModelElementContainer container) {
    if ((container instanceof GUIPlugin)) {
      GUIPlugin gp = ((GUIPlugin) container);
      EObject _function = gp.getFunction();
      Function fun = ((Function) _function);
      return fun.getOutputs().stream().collect(Collectors.<info.scce.dime.gUIPlugin.Output>toSet());
    } else {
      Set<info.scce.dime.gUIPlugin.Output> contained = new HashSet<info.scce.dime.gUIPlugin.Output>();
      EList<Container> _allContainers = container.getAllContainers();
      for (final Container c : _allContainers) {
        contained.addAll(this.findGUIPlugins(c));
      }
      return contained;
    }
  }
  
  private void checkOutput(final ProcessId outputId, final Output output, final GUIBranchPort guiBranchPort) {
    final EObject eObj = guiBranchPort.getPortNode();
    boolean _isPrimitive = GUIBranchPort.isPrimitive(eObj);
    if (_isPrimitive) {
      if ((output instanceof info.scce.dime.process.process.PrimitiveOutputPort)) {
        boolean _isList = guiBranchPort.isList();
        boolean _isIsList = ((info.scce.dime.process.process.PrimitiveOutputPort)output).isIsList();
        boolean _notEquals = (_isList != _isIsList);
        if (_notEquals) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("is List not matching. should be ");
          boolean _isList_1 = guiBranchPort.isList();
          _builder.append(_isList_1);
          this.addError(outputId, _builder.toString());
        }
        PrimitiveType _primitiveType = GUIBranchPort.getPrimitiveType(eObj);
        PrimitiveType _data = GUIBranchPort.toData(((info.scce.dime.process.process.PrimitiveOutputPort)output).getDataType());
        boolean _notEquals_1 = (!Objects.equal(_primitiveType, _data));
        if (_notEquals_1) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("primitive type not matching. should be ");
          String _string = GUIBranchPort.getPrimitiveType(eObj).toString();
          _builder_1.append(_string);
          this.addError(outputId, _builder_1.toString());
        }
      } else {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("should be primitive");
        this.addError(outputId, _builder_2.toString());
      }
    } else {
      final ComplexGUIBranchPort complexGUIBranchPort = ((ComplexGUIBranchPort) guiBranchPort);
      if ((output instanceof info.scce.dime.process.process.ComplexOutputPort)) {
        boolean _isList_2 = guiBranchPort.isList();
        boolean _isIsList_1 = ((info.scce.dime.process.process.ComplexOutputPort)output).isIsList();
        boolean _notEquals_2 = (_isList_2 != _isIsList_1);
        if (_notEquals_2) {
          StringConcatenation _builder_3 = new StringConcatenation();
          _builder_3.append("is List not matching. should be ");
          boolean _isList_3 = guiBranchPort.isList();
          _builder_3.append(_isList_3);
          this.addError(outputId, _builder_3.toString());
        }
        boolean _equals = this._dataExtension.getOriginalType(complexGUIBranchPort.getType()).equals(this._dataExtension.getOriginalType(((info.scce.dime.process.process.ComplexOutputPort)output).getDataType()));
        boolean _not = (!_equals);
        if (_not) {
          StringConcatenation _builder_4 = new StringConcatenation();
          _builder_4.append("complex type not matching. should be ");
          String _name = this._dataExtension.getOriginalType(complexGUIBranchPort.getType()).getName();
          _builder_4.append(_name);
          this.addError(outputId, _builder_4.toString());
        }
      } else {
        StringConcatenation _builder_5 = new StringConcatenation();
        _builder_5.append("should be complex");
        this.addError(outputId, _builder_5.toString());
      }
    }
  }
  
  private EObject getRelatedPort(final Iterable<EObject> col, final Object item) {
    String itemName = null;
    if ((item instanceof OutputPort)) {
      itemName = ((OutputPort) item).getName();
    }
    if ((item instanceof Input)) {
      itemName = ((Input) item).getName();
    }
    if ((item instanceof Variable)) {
      itemName = ((Variable) item).getName();
    } else {
      if ((item instanceof Attribute)) {
        if ((item instanceof PrimitiveAttribute)) {
          itemName = ((PrimitiveAttribute) item).getAttribute().getName();
        }
        if ((item instanceof ComplexAttribute)) {
          itemName = ((ComplexAttribute) item).getAttribute().getName();
        }
        if ((item instanceof ComplexListAttribute)) {
          itemName = ((ComplexListAttribute) item).getAttributeName().getLiteral();
        }
        if ((item instanceof PrimitiveListAttribute)) {
          itemName = ((PrimitiveListAttribute) item).getAttributeName().getLiteral();
        }
      }
    }
    if ((item instanceof info.scce.dime.gUIPlugin.Output)) {
      itemName = ((info.scce.dime.gUIPlugin.Output) item).getOutputName();
    }
    if ((item instanceof AbstractParameter)) {
      itemName = ((AbstractParameter) item).getName();
    }
    if ((item instanceof String)) {
      itemName = ((String) item);
    }
    if ((itemName == null)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("unknown port type of item: ");
      _builder.append(item);
      throw new IllegalStateException(_builder.toString());
    }
    for (final EObject eObj : col) {
      {
        String colItemName = null;
        if ((eObj instanceof OutputPort)) {
          colItemName = ((OutputPort) eObj).getName();
        }
        if ((eObj instanceof Input)) {
          colItemName = ((Input) eObj).getName();
        }
        if ((eObj instanceof Variable)) {
          colItemName = ((Variable) eObj).getName();
        }
        if ((eObj instanceof Attribute)) {
          if ((eObj instanceof PrimitiveAttribute)) {
            colItemName = ((PrimitiveAttribute) eObj).getAttribute().getName();
          }
          if ((eObj instanceof ComplexAttribute)) {
            colItemName = ((ComplexAttribute) eObj).getAttribute().getName();
          }
          if ((eObj instanceof ComplexListAttribute)) {
            colItemName = ((ComplexListAttribute) eObj).getAttributeName().getLiteral();
          }
          if ((eObj instanceof PrimitiveListAttribute)) {
            colItemName = ((PrimitiveListAttribute) eObj).getAttributeName().getLiteral();
          }
        }
        if ((eObj instanceof AddToSubmission)) {
          colItemName = ((AddToSubmission) eObj).getOutputName();
        }
        if ((eObj instanceof AbstractParameter)) {
          colItemName = ((AbstractParameter) eObj).getName();
        }
        if ((eObj instanceof info.scce.dime.gUIPlugin.Output)) {
          colItemName = ((info.scce.dime.gUIPlugin.Output) eObj).getOutputName();
        }
        if ((colItemName == null)) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("unknown port type of colItem: ");
          _builder_1.append(eObj);
          throw new IllegalStateException(_builder_1.toString());
        }
        boolean _equals = itemName.equals(colItemName);
        if (_equals) {
          return eObj;
        }
      }
    }
    return null;
  }
}
