/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.process.actions;

import graphmodel.Edge;
import graphmodel.Node;
import graphmodel.internal.InternalEdge;
import graphmodel.internal.InternalModelElement;
import graphmodel.internal._Point;
import info.scce.dime.api.DIMECustomAction;
import info.scce.dime.process.helper.PortUtils;
import info.scce.dime.process.process.AbstractBranch;
import info.scce.dime.process.process.BlueprintSIB;
import info.scce.dime.process.process.BranchBlueprint;
import info.scce.dime.process.process.BranchConnector;
import info.scce.dime.process.process.IO;
import info.scce.dime.process.process.Input;
import info.scce.dime.process.process.InputStatic;
import info.scce.dime.process.process.Output;
import info.scce.dime.process.process.OutputPort;
import info.scce.dime.process.process.SIB;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Action that converts a SIB to a Blueprint SIB.
 */
@SuppressWarnings("all")
public abstract class ConvertToBlueprintSIB<T extends SIB> extends DIMECustomAction<T> {
  @Extension
  private PortUtils _portUtils = new PortUtils();
  
  @Override
  public String getName() {
    return "Convert to Blueprint";
  }
  
  @Override
  public void execute(final T sib) {
    final BlueprintSIB newSib = this.convert(sib);
    final Consumer<AbstractBranch> _function = new Consumer<AbstractBranch>() {
      @Override
      public void accept(final AbstractBranch it) {
        ConvertToBlueprintSIB.this.convert(it, newSib);
      }
    };
    this._processExtension.getBranches(sib).forEach(_function);
    final EList<BranchBlueprint> branches = newSib.getBranchBlueprintSuccessors();
    int _size = branches.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      newSib.setDefaultBranch(IterableExtensions.<BranchBlueprint>head(branches));
    }
    sib.delete();
  }
  
  public BlueprintSIB convert(final T sib) {
    final Function1<SIB, Boolean> _function = new Function1<SIB, Boolean>() {
      @Override
      public Boolean apply(final SIB it) {
        return Boolean.valueOf(ConvertToBlueprintSIB.this._processExtension.hasDefaultContent(it, sib));
      }
    };
    final Set<SIB> defaultContentSIBs = IterableExtensions.<SIB>toSet(IterableExtensions.<SIB>filter(this._gUIExtension.<SIB>find(sib.getRootElement(), SIB.class), _function));
    final BlueprintSIB newSib = this.createBlueprintSIB(sib);
    this.overtakeIncomingEdgesOf(newSib, sib);
    final Consumer<Input> _function_1 = new Consumer<Input>() {
      @Override
      public void accept(final Input it) {
        ConvertToBlueprintSIB.this.convert(it, newSib);
      }
    };
    sib.getInputs().forEach(_function_1);
    final Consumer<SIB> _function_2 = new Consumer<SIB>() {
      @Override
      public void accept(final SIB it) {
        ConvertToBlueprintSIB.this._processExtension.setDefaultContent(it, newSib);
      }
    };
    defaultContentSIBs.forEach(_function_2);
    return newSib;
  }
  
  public abstract BlueprintSIB createBlueprintSIB(final T sib);
  
  public IO convert(final Input port, final BlueprintSIB newSib) {
    final IO newPort = this.createInput(port, newSib);
    this.overtakeIncomingEdgesOf(newPort, port);
    return newPort;
  }
  
  public IO createInput(final Input port, final BlueprintSIB newSib) {
    IO _switchResult = null;
    boolean _matched = false;
    if (port instanceof InputStatic) {
      _matched=true;
      _switchResult = PortUtils.addStaticPort(newSib, port);
    }
    if (!_matched) {
      _switchResult = PortUtils.addInput(newSib, port);
    }
    return _switchResult;
  }
  
  public BranchBlueprint convert(final AbstractBranch branch, final BlueprintSIB newSib) {
    final Function1<SIB, Boolean> _function = new Function1<SIB, Boolean>() {
      @Override
      public Boolean apply(final SIB it) {
        return Boolean.valueOf(ConvertToBlueprintSIB.this._processExtension.hasMajorBranch(it, branch));
      }
    };
    final Set<SIB> majorBranchSIBs = IterableExtensions.<SIB>toSet(IterableExtensions.<SIB>filter(this._gUIExtension.<SIB>find(branch.getRootElement(), SIB.class), _function));
    final BranchBlueprint newBranch = this.createBranch(branch, newSib);
    final BranchConnector newBranchConnector = newSib.newBranchConnector(newBranch);
    this.overtakeBendpointsOf(newBranchConnector, IterableExtensions.<BranchConnector>head(branch.getIncomingBranchConnectors()));
    final Consumer<OutputPort> _function_1 = new Consumer<OutputPort>() {
      @Override
      public void accept(final OutputPort it) {
        ConvertToBlueprintSIB.this.convert(it, newBranch);
      }
    };
    branch.getOutputPorts().forEach(_function_1);
    this.overtakeOutgoingEdgesOf(newBranch, branch);
    final Consumer<SIB> _function_2 = new Consumer<SIB>() {
      @Override
      public void accept(final SIB it) {
        ConvertToBlueprintSIB.this._processExtension.setMajorBranch(it, newBranch);
      }
    };
    majorBranchSIBs.forEach(_function_2);
    return newBranch;
  }
  
  public BranchBlueprint createBranch(final AbstractBranch branch, final BlueprintSIB newSib) {
    BranchBlueprint _newBranchBlueprint = newSib.getRootElement().newBranchBlueprint(branch.getX(), branch.getY(), branch.getWidth(), branch.getHeight());
    final Procedure1<BranchBlueprint> _function = new Procedure1<BranchBlueprint>() {
      @Override
      public void apply(final BranchBlueprint it) {
        it.setName(branch.getName());
      }
    };
    return ObjectExtensions.<BranchBlueprint>operator_doubleArrow(_newBranchBlueprint, _function);
  }
  
  public OutputPort convert(final Output port, final BranchBlueprint newBranch) {
    final OutputPort newPort = this.createOutput(port, newBranch);
    this.overtakeOutgoingEdgesOf(newPort, port);
    return newPort;
  }
  
  public OutputPort createOutput(final Output port, final BranchBlueprint newBranch) {
    return this._portUtils.addOutput(newBranch, port);
  }
  
  public void overtakeIncomingEdgesOf(final Node newNode, final Node node) {
    final Consumer<Edge> _function = new Consumer<Edge>() {
      @Override
      public void accept(final Edge it) {
        it.reconnectTarget(newNode);
      }
    };
    node.<Edge>getIncoming().forEach(_function);
  }
  
  public void overtakeOutgoingEdgesOf(final Node newNode, final Node node) {
    final Consumer<Edge> _function = new Consumer<Edge>() {
      @Override
      public void accept(final Edge it) {
        it.reconnectSource(newNode);
      }
    };
    node.<Edge>getOutgoing().forEach(_function);
  }
  
  public void overtakeBendpointsOf(final Edge newEdge, final Edge edge) {
    InternalModelElement _internalElement_ = edge.getInternalElement_();
    EList<_Point> _bendpoints = ((InternalEdge) _internalElement_).getBendpoints();
    for (final _Point bp : _bendpoints) {
      newEdge.addBendpoint(bp.getX(), bp.getY());
    }
  }
}
