/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.process.actions;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import graphmodel.Edge;
import graphmodel.IdentifiableElement;
import info.scce.dime.api.DIMECustomAction;
import info.scce.dime.process.build.ProcessSIBBuild;
import info.scce.dime.process.hooks.DeleteBranchesOnSIBDelete;
import info.scce.dime.process.process.BranchConnector;
import info.scce.dime.process.process.ControlFlow;
import info.scce.dime.process.process.Input;
import info.scce.dime.process.process.ProcessSIB;
import info.scce.dime.process.process.SIB;
import java.util.ArrayList;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtext.xbase.lib.Functions.Function0;

/**
 * Offers basic conversion between sibs. Call the <code>convertSIB()</code> in your implementation of the <code>execute()</code> method. You will get back
 * an instance of your target sib type from the c-api and can add some specific synchronization/copying code.
 */
@SuppressWarnings("all")
public abstract class ConvertSIBCustomAction<T extends IdentifiableElement> extends DIMECustomAction<T> {
  /**
   * Converts a sib to another sib type. It reuses the inputs (including static inputs with their values)
   * branches, their outputs, and the connected data flow edges to and from inputs and outputs.
   * 
   * @param sib the sib to convert.
   * @param createNewSIB a closure for instantiating the new sib.
   */
  public <U extends SIB> U convertSIB(final SIB sib, final Function0<? extends U> createNewSIB) {
    final SIB cSIB = sib;
    U _xblockexpression = null;
    {
      final U newProcessSIB = createNewSIB.apply();
      _xblockexpression = ((U) newProcessSIB);
    }
    final U cNewSIB = _xblockexpression;
    new DeleteBranchesOnSIBDelete().preDelete(cNewSIB);
    final Consumer<BranchConnector> _function = new Consumer<BranchConnector>() {
      @Override
      public void accept(final BranchConnector it) {
        it.reconnectSource(cNewSIB);
      }
    };
    Iterables.<BranchConnector>filter(Lists.<Edge>newArrayList(cSIB.getOutgoing()), BranchConnector.class).forEach(_function);
    final Consumer<ControlFlow> _function_1 = new Consumer<ControlFlow>() {
      @Override
      public void accept(final ControlFlow it) {
        it.reconnectTarget(cNewSIB);
      }
    };
    Iterables.<ControlFlow>filter(Lists.<ControlFlow>newArrayList(cSIB.getIncoming()), ControlFlow.class).forEach(_function_1);
    final Consumer<Input> _function_2 = new Consumer<Input>() {
      @Override
      public void accept(final Input it) {
        it.delete();
      }
    };
    Lists.<Input>newArrayList(cNewSIB.getInputs()).forEach(_function_2);
    EList<Input> _inputs = cSIB.getInputs();
    final Consumer<Input> _function_3 = new Consumer<Input>() {
      @Override
      public void accept(final Input it) {
        it.moveTo(cNewSIB, it.getX(), it.getY());
      }
    };
    new ArrayList<Input>(_inputs).forEach(_function_3);
    cNewSIB.setLabel(cSIB.getLabel());
    cNewSIB.setName(cSIB.getName());
    ProcessSIBBuild.update(((ProcessSIB) cNewSIB));
    cSIB.delete();
    return cNewSIB;
  }
}
