/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.gui.hooks;

import com.google.common.collect.Iterables;
import graphmodel.ModelElementContainer;
import graphmodel.Node;
import info.scce.dime.gui.gui.Argument;
import info.scce.dime.gui.gui.ComplexOutputPort;
import info.scce.dime.gui.gui.ComplexVariable;
import info.scce.dime.gui.gui.Event;
import info.scce.dime.gui.gui.ExtensionContext;
import info.scce.dime.gui.gui.GUI;
import info.scce.dime.gui.gui.GUISIB;
import info.scce.dime.gui.gui.OutputPort;
import info.scce.dime.gui.gui.Placeholder;
import info.scce.dime.gui.gui.PrimitiveOutputPort;
import info.scce.dime.gui.gui.PrimitiveVariable;
import info.scce.dime.gui.gui.SIB;
import info.scce.dime.gui.gui.Variable;
import info.scce.dime.gui.helper.ComplexGUIBranchPort;
import info.scce.dime.gui.helper.GUIBranch;
import info.scce.dime.gui.helper.GUIBranchPort;
import info.scce.dime.gui.helper.PrimitiveGUIBranchPort;
import info.scce.dime.gui.hooks.AbstractPostCreateSIBHook;
import info.scce.dime.gui.hooks.CreateComponentHook;
import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;

/**
 * The GUI SIB hook is used create the corresponding ports and branches
 * for the new GUI SIB
 * @author zweihoff
 */
@SuppressWarnings("all")
public class GUISIBHook extends AbstractPostCreateSIBHook<GUISIB> {
  private final int OFFSET = 30;
  
  /**
   * Creates the corresponding GUI SIB input ports for every input variable
   * of the referenced GUI model.
   * For every button and branch of embedded SIBs, a branch with the corresponding ports is added.
   */
  @Override
  public void postCreate(final GUISIB sib) {
    GUI lcGUI = sib.getGui();
    Iterable<Variable> inputs = this._gUIExtension.inputVariables(lcGUI);
    final Collection<GUIBranch> branches = this._gUIExtension.getGUIBranchesMerged(lcGUI);
    Iterable<Event> events = this._gUIExtension.events(lcGUI);
    sib.setLabel(lcGUI.getTitle());
    this.init(sib, lcGUI.getTitle(), Iterables.size(inputs));
    final Consumer<Variable> _function = new Consumer<Variable>() {
      @Override
      public void accept(final Variable v) {
        GUISIBHook.this.addGUIInput(v);
      }
    };
    inputs.forEach(_function);
    final Consumer<GUIBranch> _function_1 = new Consumer<GUIBranch>() {
      @Override
      public void accept(final GUIBranch it) {
        GUISIBHook.this.addBranch(it);
      }
    };
    branches.forEach(_function_1);
    final Consumer<Event> _function_2 = new Consumer<Event>() {
      @Override
      public void accept(final Event e) {
        GUISIBHook.this.addEventBranches(e.getName(), e.getOutputPorts());
      }
    };
    events.forEach(_function_2);
    this.finish();
    List<Placeholder> _find = this._gUIExtension.<Placeholder>find(lcGUI, Placeholder.class);
    for (final Placeholder ph : _find) {
      this.addArgument(sib, ph.getName());
    }
    this.layout(sib);
    ModelElementContainer _container = sib.getContainer();
    if ((_container instanceof ExtensionContext)) {
      ModelElementContainer _container_1 = sib.getContainer();
      ExtensionContext eContext = ((ExtensionContext) _container_1);
      this.layoutContext(eContext);
    }
  }
  
  public void layoutContext(final ExtensionContext node) {
    int maxHeight = this.maxHeight(node);
    node.setHeight((maxHeight + this.OFFSET));
  }
  
  public int maxHeight(final ExtensionContext context) {
    int maxLowerBound = 50;
    EList<? extends Node> _nodes = context.getNodes();
    for (final Node node : _nodes) {
      int _y = node.getY();
      int _height = node.getHeight();
      int _plus = (_y + _height);
      boolean _greaterThan = (_plus > maxLowerBound);
      if (_greaterThan) {
        int _y_1 = node.getY();
        int _height_1 = node.getHeight();
        int _plus_1 = (_y_1 + _height_1);
        maxLowerBound = _plus_1;
      }
    }
    return maxLowerBound;
  }
  
  public void layout(final SIB sib) {
    CreateComponentHook cch = new CreateComponentHook();
    cch.postCreate(sib);
  }
  
  public void addArgument(final GUISIB csib, final String name) {
    Argument ca = ((GUISIB) csib).newArgument(1, csib.getHeight());
    ca.setBlockName(name);
  }
  
  public void addGUIInput(final Variable var) {
    if ((var instanceof ComplexVariable)) {
      ComplexVariable cVar = ((ComplexVariable) var);
      this.addComplexInputPort(cVar.getName(), cVar.getDataType(), cVar.isIsList());
    } else {
      if ((var instanceof PrimitiveVariable)) {
        PrimitiveVariable pVar = ((PrimitiveVariable) var);
        this.addPrimitiveInputPort(pVar.getName(), pVar.getDataType(), pVar.isIsList());
      } else {
        throw new IllegalStateException("else in exhaustive if should not happen; please fix broken implementation");
      }
    }
  }
  
  public void addEventBranches(final String name, final List<OutputPort> outputPorts) {
    this.newEventListener(name, outputPorts.size());
    for (final OutputPort port : outputPorts) {
      {
        if ((port instanceof ComplexOutputPort)) {
          this.addEventComplexInputPort(((ComplexOutputPort)port).getName(), ((ComplexOutputPort) port).getDataType(), ((ComplexOutputPort)port).isIsList());
        }
        if ((port instanceof PrimitiveOutputPort)) {
          PrimitiveOutputPort pVar = ((PrimitiveOutputPort) port);
          this.addEventPrimitiveInputPort(pVar.getName(), pVar.getDataType(), pVar.isIsList());
        }
      }
    }
  }
  
  public void addBranch(final GUIBranch branch) {
    this.newBranch(branch.getName(), branch.getPorts().size());
    List<GUIBranchPort> _ports = branch.getPorts();
    for (final GUIBranchPort branchPort : _ports) {
      if ((branchPort instanceof ComplexGUIBranchPort)) {
        this.addComplexOutputPort(((ComplexGUIBranchPort)branchPort).getName(), ((ComplexGUIBranchPort)branchPort).getType(), ((ComplexGUIBranchPort)branchPort).isList());
      } else {
        if ((branchPort instanceof PrimitiveGUIBranchPort)) {
          this.addPrimitiveOutputPort(((PrimitiveGUIBranchPort)branchPort).getName(), this._gUIExtension.toGUI(((PrimitiveGUIBranchPort)branchPort).getType()), ((PrimitiveGUIBranchPort)branchPort).isList());
        }
      }
    }
  }
}
