/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.gui.helper;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import de.jabc.cinco.meta.core.referenceregistry.ReferenceRegistry;
import de.jabc.cinco.meta.core.utils.registry.NonEmptyRegistry;
import de.jabc.cinco.meta.runtime.xapi.FileExtension;
import de.jabc.cinco.meta.runtime.xapi.ResourceExtension;
import de.jabc.cinco.meta.runtime.xapi.WorkspaceExtension;
import graphmodel.Container;
import graphmodel.GraphModel;
import graphmodel.IdentifiableElement;
import graphmodel.ModelElement;
import graphmodel.ModelElementContainer;
import graphmodel.Node;
import info.scce.dime.api.DIMEGraphModelExtension;
import info.scce.dime.data.data.PrimitiveType;
import info.scce.dime.data.data.Type;
import info.scce.dime.data.helper.DataExtension;
import info.scce.dime.data.helper.JavaIdentifierUtils;
import info.scce.dime.gUIPlugin.ComplexInputParameter;
import info.scce.dime.gUIPlugin.ComplexParameter;
import info.scce.dime.gui.gui.Attribute;
import info.scce.dime.gui.gui.BooleanInputStatic;
import info.scce.dime.gui.gui.Branch;
import info.scce.dime.gui.gui.Button;
import info.scce.dime.gui.gui.ButtonOptions;
import info.scce.dime.gui.gui.ButtonToolbar;
import info.scce.dime.gui.gui.ButtonToolbarType;
import info.scce.dime.gui.gui.ComplexAttribute;
import info.scce.dime.gui.gui.ComplexAttributeConnector;
import info.scce.dime.gui.gui.ComplexListAttribute;
import info.scce.dime.gui.gui.ComplexListAttributeConnector;
import info.scce.dime.gui.gui.ComplexOutputPort;
import info.scce.dime.gui.gui.ComplexVariable;
import info.scce.dime.gui.gui.DataContext;
import info.scce.dime.gui.gui.DispatchedGUISIB;
import info.scce.dime.gui.gui.Event;
import info.scce.dime.gui.gui.ExtensionContext;
import info.scce.dime.gui.gui.Field;
import info.scce.dime.gui.gui.Form;
import info.scce.dime.gui.gui.GUI;
import info.scce.dime.gui.gui.GUIPlugin;
import info.scce.dime.gui.gui.GUISIB;
import info.scce.dime.gui.gui.GUISIBModal;
import info.scce.dime.gui.gui.IO;
import info.scce.dime.gui.gui.IS;
import info.scce.dime.gui.gui.Input;
import info.scce.dime.gui.gui.InputPort;
import info.scce.dime.gui.gui.InputStatic;
import info.scce.dime.gui.gui.InputType;
import info.scce.dime.gui.gui.IntegerInputStatic;
import info.scce.dime.gui.gui.Iteration;
import info.scce.dime.gui.gui.LinkSIB;
import info.scce.dime.gui.gui.ListenerContext;
import info.scce.dime.gui.gui.ModalCloseButton;
import info.scce.dime.gui.gui.PrimitiveAttribute;
import info.scce.dime.gui.gui.PrimitiveListAttribute;
import info.scce.dime.gui.gui.PrimitiveVariable;
import info.scce.dime.gui.gui.ProcessSIB;
import info.scce.dime.gui.gui.RealInputStatic;
import info.scce.dime.gui.gui.TableLoad;
import info.scce.dime.gui.gui.TextInputStatic;
import info.scce.dime.gui.gui.TimestampInputStatic;
import info.scce.dime.gui.gui.Variable;
import info.scce.dime.gui.helper.GUIBranch;
import info.scce.dime.gui.helper.GUIBranchPort;
import info.scce.dime.process.process.AtomicSIB;
import info.scce.dime.process.process.ComplexInputPort;
import info.scce.dime.process.process.EndSIB;
import info.scce.dime.process.process.PrimitiveInputPort;
import info.scce.dime.process.process.PrimitiveOutputPort;
import info.scce.dime.process.process.RemoveFromListSIB;
import info.scce.dime.process.process.RetrieveCurrentUserSIB;
import info.scce.dime.process.process.RetrieveOfTypeSIB;
import info.scce.dime.process.process.SIB;
import java.io.File;
import java.net.URI;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class GUIExtension extends DIMEGraphModelExtension {
  @Extension
  private static WorkspaceExtension _workspaceExtension = new WorkspaceExtension();
  
  @Extension
  protected DataExtension _dataExtension = DataExtension.getInstance();
  
  @Extension
  protected ResourceExtension _resourceExtension = new ResourceExtension();
  
  @Extension
  protected FileExtension _fileExtension = new FileExtension();
  
  private boolean cacheValues = false;
  
  private final NonEmptyRegistry<ModelElementContainer, Map<Class<? extends IdentifiableElement>, List<? extends IdentifiableElement>>> _modelElements = new NonEmptyRegistry<ModelElementContainer, Map<Class<? extends IdentifiableElement>, List<? extends IdentifiableElement>>>(new Function<ModelElementContainer, Map<Class<? extends IdentifiableElement>, List<? extends IdentifiableElement>>>() {
    @Override
    public Map<Class<? extends IdentifiableElement>, List<? extends IdentifiableElement>> apply(final ModelElementContainer it) {
      return CollectionLiterals.<Class<? extends IdentifiableElement>, List<? extends IdentifiableElement>>newHashMap();
    }
  });
  
  private final NonEmptyRegistry<ModelElement, Map<Class<?>, ModelElement>> _firstParents = new NonEmptyRegistry<ModelElement, Map<Class<?>, ModelElement>>(new Function<ModelElement, Map<Class<?>, ModelElement>>() {
    @Override
    public Map<Class<?>, ModelElement> apply(final ModelElement it) {
      return CollectionLiterals.<Class<?>, ModelElement>newHashMap();
    }
  });
  
  private Map<GUI, Set<GUI>> _subGUIs;
  
  private Map<GUI, List<GUI>> _superGUIs;
  
  private final HashMap<ModelElementContainer, Collection<GUIBranch>> _branchElements = CollectionLiterals.<ModelElementContainer, Collection<GUIBranch>>newHashMap();
  
  private final HashMap<ModelElementContainer, Collection<GUIBranch>> _processBranchElements = CollectionLiterals.<ModelElementContainer, Collection<GUIBranch>>newHashMap();
  
  private final HashMap<ModelElementContainer, Set<info.scce.dime.process.process.Process>> _embeddedProcesses = CollectionLiterals.<ModelElementContainer, Set<info.scce.dime.process.process.Process>>newHashMap();
  
  private final HashMap<ModelElement, Node> _surroundingLoops = CollectionLiterals.<ModelElement, Node>newHashMap();
  
  private final HashMap<Container, List<Input>> _formInputs = CollectionLiterals.<Container, List<Input>>newHashMap();
  
  private final Function1<? super IdentifiableElement, ? extends ModelElementContainer> digIntoGUISIBs = new Function1<IdentifiableElement, ModelElementContainer>() {
    @Override
    public ModelElementContainer apply(final IdentifiableElement it) {
      GUI _switchResult = null;
      boolean _matched = false;
      if (it instanceof GUISIB) {
        _matched=true;
        _switchResult = ((GUISIB)it).getGui();
      }
      return _switchResult;
    }
  };
  
  private final Function1<? super IdentifiableElement, ? extends ModelElementContainer> digIntoNonExtendingGUISIBs = new Function1<IdentifiableElement, ModelElementContainer>() {
    @Override
    public ModelElementContainer apply(final IdentifiableElement it) {
      GUISIB _switchResult = null;
      boolean _matched = false;
      if (it instanceof GUISIB) {
        boolean _hasParent = GUIExtension.this.<ExtensionContext>hasParent(((ModelElement)it), ExtensionContext.class);
        boolean _not = (!_hasParent);
        if (_not) {
          _matched=true;
          _switchResult = ((GUISIB)it);
        }
      }
      return _switchResult;
    }
  };
  
  public static long lastModified2(final GUIPlugin g) {
    URI _locationURI = GUIExtension._workspaceExtension.getFile(g).getLocationURI();
    return new File(_locationURI).lastModified();
  }
  
  public static long lastModified2(final GraphModel g) {
    URI _locationURI = GUIExtension._workspaceExtension.getFile(g).getLocationURI();
    return new File(_locationURI).lastModified();
  }
  
  public String graphModelFileName(final GraphModel g) {
    return GUIExtension._workspaceExtension.getFile(g).getLocationURI().toString();
  }
  
  public static Long lastModified2(final Iterable<GraphModel> gs) {
    final Function1<GraphModel, Long> _function = new Function1<GraphModel, Long>() {
      @Override
      public Long apply(final GraphModel it) {
        return Long.valueOf(GUIExtension.lastModified2(it));
      }
    };
    return IterableExtensions.<Long>max(IterableExtensions.<GraphModel, Long>map(gs, _function));
  }
  
  public GUIExtension() {
    this(false);
  }
  
  public GUIExtension(final boolean cacheValues) {
    this.cacheValues = true;
  }
  
  public DataExtension dataExtension() {
    return this._dataExtension;
  }
  
  /**
   * Maps a primitive type (i.e. the enum value) of native ui components to
   * primitive types of data models.
   * 
   * @param it the primitive type enum value of gui.
   * @return the corresponding primitive type enum value of data.
   */
  public PrimitiveType toData(final info.scce.dime.gUIPlugin.PrimitiveType it) {
    PrimitiveType _switchResult = null;
    if (it != null) {
      switch (it) {
        case BOOLEAN:
          _switchResult = PrimitiveType.BOOLEAN;
          break;
        case INTEGER:
          _switchResult = PrimitiveType.INTEGER;
          break;
        case REAL:
          _switchResult = PrimitiveType.REAL;
          break;
        case TEXT:
          _switchResult = PrimitiveType.TEXT;
          break;
        case TIMESTAMP:
          _switchResult = PrimitiveType.TIMESTAMP;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  /**
   * Maps a primitive type (i.e. the enum value) of native ui components to
   * primitive types of data models.
   * 
   * @param it the primitive type enum value of gui.
   * @return the corresponding primitive type enum value of data.
   */
  public info.scce.dime.gui.gui.PrimitiveType toGUI(final PrimitiveType it) {
    info.scce.dime.gui.gui.PrimitiveType _switchResult = null;
    if (it != null) {
      switch (it) {
        case BOOLEAN:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.BOOLEAN;
          break;
        case INTEGER:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.INTEGER;
          break;
        case REAL:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.REAL;
          break;
        case TEXT:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.TEXT;
          break;
        case TIMESTAMP:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.TIMESTAMP;
          break;
        case FILE:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.FILE;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  /**
   * Maps a primitive type (i.e. the enum value) of data models to
   * primitive types of process models.
   * 
   * @param it the primitive type enum value of data.
   * @return the corresponding primitive type enum value of process.
   */
  public info.scce.dime.process.process.PrimitiveType toProcess(final PrimitiveType it) {
    info.scce.dime.process.process.PrimitiveType _switchResult = null;
    if (it != null) {
      switch (it) {
        case BOOLEAN:
          _switchResult = info.scce.dime.process.process.PrimitiveType.BOOLEAN;
          break;
        case FILE:
          _switchResult = info.scce.dime.process.process.PrimitiveType.FILE;
          break;
        case INTEGER:
          _switchResult = info.scce.dime.process.process.PrimitiveType.INTEGER;
          break;
        case REAL:
          _switchResult = info.scce.dime.process.process.PrimitiveType.REAL;
          break;
        case TEXT:
          _switchResult = info.scce.dime.process.process.PrimitiveType.TEXT;
          break;
        case TIMESTAMP:
          _switchResult = info.scce.dime.process.process.PrimitiveType.TIMESTAMP;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  protected String _defaultValue(final info.scce.dime.gui.gui.PrimitiveType pt) {
    return this.defaultValue(this._dataExtension.toData(pt));
  }
  
  protected String _defaultValue(final PrimitiveType pt) {
    if (pt != null) {
      switch (pt) {
        case BOOLEAN:
          return "false";
        case FILE:
          return "null";
        case INTEGER:
          return "0";
        case REAL:
          return "0.0";
        case TEXT:
          return "\'\'";
        case TIMESTAMP:
          return "null";
        default:
          break;
      }
    }
    return null;
  }
  
  public String primitiveType(final PrimitiveType pt) {
    if (pt != null) {
      switch (pt) {
        case BOOLEAN:
          return "bool";
        case FILE:
          return "FileReference";
        case INTEGER:
          return "int";
        case REAL:
          return "double";
        case TEXT:
          return "String";
        case TIMESTAMP:
          return "DateTime";
        default:
          break;
      }
    }
    return null;
  }
  
  public boolean getIsDateTime(final Field field) {
    return ((((Objects.equal(field.getInputType(), InputType.DATE) || 
      Objects.equal(field.getInputType(), InputType.DATETIME)) || 
      Objects.equal(field.getInputType(), InputType.MONTH)) || 
      Objects.equal(field.getInputType(), InputType.TIME)) || 
      Objects.equal(field.getInputType(), InputType.WEEK));
  }
  
  public boolean getIsFile(final Field field) {
    return (Objects.equal(field.getInputType(), InputType.SIMPLE_FILE) || 
      Objects.equal(field.getInputType(), InputType.ADVANCED_FILE));
  }
  
  public String defaultValue(final PrimitiveType pt, final boolean list) {
    String _xifexpression = null;
    if (list) {
      _xifexpression = "new DIMEList()";
    } else {
      _xifexpression = this.defaultValue(pt);
    }
    return _xifexpression;
  }
  
  protected String _defaultValue(final PrimitiveVariable pv) {
    String _xblockexpression = null;
    {
      boolean _isIsList = pv.isIsList();
      if (_isIsList) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new DIMEList()");
        return _builder.toString();
      }
      _xblockexpression = this.defaultValue(this._dataExtension.toData(pv.getDataType()));
    }
    return _xblockexpression;
  }
  
  protected String _defaultValue(final PrimitiveInputPort pv) {
    String _xblockexpression = null;
    {
      boolean _isIsList = pv.isIsList();
      if (_isIsList) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("const []");
        return _builder.toString();
      }
      _xblockexpression = this.defaultValue(this._dataExtension.toData(pv.getDataType()));
    }
    return _xblockexpression;
  }
  
  protected String _defaultValue(final ComplexInputPort pv) {
    String _xblockexpression = null;
    {
      boolean _isIsList = pv.isIsList();
      if (_isIsList) {
        StringConcatenation _builder = new StringConcatenation();
        String _escapeDart = JavaIdentifierUtils.escapeDart(pv.getDataType().getRootElement().getModelName());
        _builder.append(_escapeDart);
        final String data = _builder.toString();
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(data);
        _builder_1.append(".");
        _builder_1.append(data);
        _builder_1.append("CastUtil.newList");
        String _firstUpper = StringExtensions.toFirstUpper(JavaIdentifierUtils.escapeDart(pv.getDataType().getName()));
        _builder_1.append(_firstUpper);
        _builder_1.append("()");
        return _builder_1.toString();
      }
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("null");
      _xblockexpression = _builder_2.toString();
    }
    return _xblockexpression;
  }
  
  public CharSequence variableListStatus(final CharSequence s, final Variable v) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isIsList = v.isIsList();
      if (_isIsList) {
        _builder.append("DIMEList<");
      }
    }
    _builder.append(s);
    {
      boolean _isIsList_1 = v.isIsList();
      if (_isIsList_1) {
        _builder.append(">");
      }
    }
    return _builder;
  }
  
  public CharSequence variableType(final PrimitiveVariable v) {
    return this.variableListStatus(this.primitiveType(this._dataExtension.toData(v.getDataType())), v);
  }
  
  protected String _defaultValue(final ComplexVariable pt) {
    boolean _isIsList = pt.isIsList();
    if (_isIsList) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("const []");
      return _builder.toString();
    }
    StringConcatenation _builder_1 = new StringConcatenation();
    _builder_1.append("null");
    return _builder_1.toString();
  }
  
  protected String _defaultValue(final InputStatic input) {
    return this.staticValue(input);
  }
  
  public String staticValue(final InputStatic input) {
    boolean _matched = false;
    if (input instanceof TextInputStatic) {
      _matched=true;
      String _value = null;
      if (((TextInputStatic)input)!=null) {
        _value=((TextInputStatic)input).getValue();
      }
      boolean _tripleEquals = (_value == null);
      if (_tripleEquals) {
        System.err.println(("Value is null for input: " + input));
      }
      String _replaceAll = ((TextInputStatic)input).getValue().replaceAll("\\{\\{", "\'+").replaceAll("\\}\\}", ".toString()+\'");
      String _plus = ("\'" + _replaceAll);
      return (_plus + "\'");
    }
    if (!_matched) {
      if (input instanceof RealInputStatic) {
        _matched=true;
        return Double.valueOf(((RealInputStatic)input).getValue()).toString();
      }
    }
    if (!_matched) {
      if (input instanceof IntegerInputStatic) {
        _matched=true;
        return Long.valueOf(((IntegerInputStatic)input).getValue()).toString();
      }
    }
    if (!_matched) {
      if (input instanceof BooleanInputStatic) {
        _matched=true;
        return Boolean.valueOf(((BooleanInputStatic)input).isValue()).toString();
      }
    }
    if (!_matched) {
      if (input instanceof TimestampInputStatic) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("staticDate");
        String _escapeDart = JavaIdentifierUtils.escapeDart(((TimestampInputStatic)input).getId());
        _builder.append(_escapeDart);
        return _builder.toString();
      }
    }
    return "\'\'";
  }
  
  public String staticValue(final info.scce.dime.process.process.InputStatic input) {
    boolean _matched = false;
    if (input instanceof info.scce.dime.process.process.TextInputStatic) {
      _matched=true;
      String _value = ((info.scce.dime.process.process.TextInputStatic)input).getValue();
      String _plus = ("\'" + _value);
      return (_plus + "\'");
    }
    if (!_matched) {
      if (input instanceof info.scce.dime.process.process.RealInputStatic) {
        _matched=true;
        return Double.valueOf(((info.scce.dime.process.process.RealInputStatic)input).getValue()).toString();
      }
    }
    if (!_matched) {
      if (input instanceof info.scce.dime.process.process.IntegerInputStatic) {
        _matched=true;
        return Long.valueOf(((info.scce.dime.process.process.IntegerInputStatic)input).getValue()).toString();
      }
    }
    if (!_matched) {
      if (input instanceof info.scce.dime.process.process.BooleanInputStatic) {
        _matched=true;
        return Boolean.valueOf(((info.scce.dime.process.process.BooleanInputStatic)input).isValue()).toString();
      }
    }
    if (!_matched) {
      if (input instanceof info.scce.dime.process.process.TimestampInputStatic) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("getStaticDate(");
        String _string = Long.valueOf(((info.scce.dime.process.process.TimestampInputStatic)input).getValue()).toString();
        _builder.append(_string);
        _builder.append("000)");
        return _builder.toString();
      }
    }
    return "\'\'";
  }
  
  protected String _defaultValue(final IO input) {
    if ((input instanceof info.scce.dime.gui.gui.ComplexInputPort)) {
      boolean _isIsList = ((info.scce.dime.gui.gui.ComplexInputPort)input).isIsList();
      if (_isIsList) {
        StringConcatenation _builder = new StringConcatenation();
        String _escapeDart = JavaIdentifierUtils.escapeDart(((info.scce.dime.gui.gui.ComplexInputPort)input).getDataType().getRootElement().getModelName());
        _builder.append(_escapeDart);
        final String data = _builder.toString();
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(data);
        _builder_1.append(".");
        _builder_1.append(data);
        _builder_1.append("CastUtil.newList");
        String _firstUpper = StringExtensions.toFirstUpper(JavaIdentifierUtils.escapeDart(((info.scce.dime.gui.gui.ComplexInputPort)input).getDataType().getName()));
        _builder_1.append(_firstUpper);
        _builder_1.append("()");
        return _builder_1.toString();
      }
      return "null";
    }
    if ((input instanceof InputStatic)) {
      return this.staticValue(((InputStatic)input));
    }
    info.scce.dime.gui.gui.PrimitiveType _dataType = ((info.scce.dime.gui.gui.PrimitiveInputPort) input).getDataType();
    if (_dataType != null) {
      switch (_dataType) {
        case BOOLEAN:
          return "false";
        case FILE:
          return "null";
        case INTEGER:
          return "0";
        case REAL:
          return "0.0";
        case TEXT:
          return "\'\'";
        case TIMESTAMP:
          return "null";
        default:
          break;
      }
    }
    return null;
  }
  
  protected String _defaultValue(final info.scce.dime.process.process.InputStatic input) {
    return this.staticValue(input);
  }
  
  /**
   * Checks whether the type of the source node (i.e., a {@link Variable} or {@link Attribute})
   * matches (i.e., is a sub type) the type of the target element (i.e., an {@link InputPort}). If not,
   * it checks whether all {@link IS}-edges pointing to any containing element on the path to the
   * root (starting from the target element itself) match the type of the target element. The latter check
   * can be seen as a smart cast for {@link IS}-edges.
   * 
   * @param source a pair associating the source node to its type. The source node has type {@link Node},
   * 		since this is the lowest common supert type of {@link Variable} or {@link Attribute}.
   * @param target a pair associating the target element to its type.
   * @return whether the type of the source node matches the type of the target element respecting
   * 		smart cast capabilities of {@link IS}-edges.
   */
  public boolean isSubTypeWithSmartCast(final Pair<Node, Type> source, final Pair<InputPort, Type> target) {
    boolean _xblockexpression = false;
    {
      final Node sourceNode = source.getKey();
      final Type sourceType = source.getValue();
      final InputPort targetElement = target.getKey();
      final Type targetType = target.getValue();
      boolean _xifexpression = false;
      boolean _isTypeOf = this._dataExtension.isTypeOf(sourceType, targetType);
      boolean _not = (!_isTypeOf);
      if (_not) {
        boolean _xblockexpression_1 = false;
        {
          final Set<ModelElement> pathToRoot = IterableExtensions.<ModelElement>toSet(this.getPathToRoot(targetElement));
          final Function1<IS, Boolean> _function = new Function1<IS, Boolean>() {
            @Override
            public Boolean apply(final IS isEdge) {
              return Boolean.valueOf(pathToRoot.contains(isEdge.getTargetElement()));
            }
          };
          final Set<IS> matchingISEdges = IterableExtensions.<IS>toSet(IterableExtensions.<IS>filter(sourceNode.<IS>getOutgoing(IS.class), _function));
          _xblockexpression_1 = (!(matchingISEdges.isEmpty() || (!IterableExtensions.<IS>forall(matchingISEdges, new Function1<IS, Boolean>() {
            @Override
            public Boolean apply(final IS isEdge) {
              boolean _xblockexpression = false;
              {
                final Type matchingType = GUIExtension.this.getType(isEdge);
                boolean _xifexpression = false;
                boolean _isNegate = isEdge.isNegate();
                boolean _not = (!_isNegate);
                if (_not) {
                  boolean _xifexpression_1 = false;
                  boolean _isExactMatch = isEdge.isExactMatch();
                  if (_isExactMatch) {
                    _xifexpression_1 = GUIExtension.this.operator_equals(matchingType, targetType);
                  } else {
                    _xifexpression_1 = GUIExtension.this._dataExtension.isTypeOf(matchingType, targetType);
                  }
                  _xifexpression = _xifexpression_1;
                } else {
                  boolean _xifexpression_2 = false;
                  boolean _isExactMatch_1 = isEdge.isExactMatch();
                  if (_isExactMatch_1) {
                    boolean _equals = GUIExtension.this.operator_equals(matchingType, targetType);
                    _xifexpression_2 = (!_equals);
                  } else {
                    boolean _isTypeOf = GUIExtension.this._dataExtension.isTypeOf(matchingType, targetType);
                    _xifexpression_2 = (!_isTypeOf);
                  }
                  _xifexpression = _xifexpression_2;
                }
                _xblockexpression = _xifexpression;
              }
              return Boolean.valueOf(_xblockexpression);
            }
          }))));
        }
        _xifexpression = _xblockexpression_1;
      } else {
        _xifexpression = true;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Retrieves the type of an {@link IS}-edge.
   * 
   * @param the {@link IS}-edge.
   * @return the type of the {@link IS}-edge.
   */
  public Type getType(final IS isEdge) {
    Type _xblockexpression = null;
    {
      final Node it = isEdge.getSourceElement();
      Type _xifexpression = null;
      if ((it instanceof ComplexVariable)) {
        final Function1<Type, Boolean> _function = new Function1<Type, Boolean>() {
          @Override
          public Boolean apply(final Type it) {
            String _id = it.getId();
            String _matchingType = isEdge.getMatchingType();
            return Boolean.valueOf(Objects.equal(_id, _matchingType));
          }
        };
        _xifexpression = IterableExtensions.<Type>findFirst(((ComplexVariable)it).getDataType().getRootElement().getTypes(), _function);
      } else {
        Type _xifexpression_1 = null;
        if ((it instanceof ComplexAttribute)) {
          final Function1<Type, Boolean> _function_1 = new Function1<Type, Boolean>() {
            @Override
            public Boolean apply(final Type it) {
              String _id = it.getId();
              String _matchingType = isEdge.getMatchingType();
              return Boolean.valueOf(Objects.equal(_id, _matchingType));
            }
          };
          _xifexpression_1 = IterableExtensions.<Type>findFirst(((ComplexAttribute)it).getAttribute().getRootElement().getTypes(), _function_1);
        } else {
          String _simpleName = it.getClass().getSimpleName();
          String _plus = (("IS-edge not starting from a complex variable or " + "attribute, but \'") + _simpleName);
          String _plus_1 = (_plus + "\'.");
          throw new IllegalStateException(_plus_1);
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Converts the GUI plug in primitive types to GUI model primitive types
   * @param pt
   * @return
   */
  public info.scce.dime.gui.gui.PrimitiveType toPrimitiveType(final info.scce.dime.gUIPlugin.PrimitiveType pt) {
    info.scce.dime.gui.gui.PrimitiveType _switchResult = null;
    if (pt != null) {
      switch (pt) {
        case BOOLEAN:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.BOOLEAN;
          break;
        case INTEGER:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.INTEGER;
          break;
        case TEXT:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.TEXT;
          break;
        case REAL:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.REAL;
          break;
        case TIMESTAMP:
          _switchResult = info.scce.dime.gui.gui.PrimitiveType.TIMESTAMP;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  /**
   * Retrieves the data type that is referenced by the specified complex parameter
   * of a GUI plugin.
   * 
   * @param p  The complex parameter that references the data type to be retrieved.
   * @return  The Type that is referenced by the specified complex parameter, or
   *   {@code null} if not existent or the data model does not exist or cannot be
   *   accessed for whatever reason.
   */
  public Type toComplexType(final ComplexOutputPort p) {
    return p.getDataType();
  }
  
  /**
   * Checks if the data node is primitive
   * 
   * @param node  A node from a data context of a GUI or Process model
   * @return if the given node is a primitive variable or attribute
   */
  public boolean isPrimitve(final Node node) {
    boolean _xblockexpression = false;
    {
      if ((node instanceof PrimitiveAttribute)) {
        return true;
      }
      if ((node instanceof info.scce.dime.process.process.PrimitiveAttribute)) {
        return true;
      }
      if ((node instanceof PrimitiveVariable)) {
        return true;
      }
      if ((node instanceof info.scce.dime.process.process.PrimitiveVariable)) {
        return true;
      }
      if ((node instanceof PrimitiveOutputPort)) {
        return true;
      }
      if ((node instanceof PrimitiveInputPort)) {
        return true;
      }
      if ((node instanceof info.scce.dime.gui.gui.PrimitiveOutputPort)) {
        return true;
      }
      if ((node instanceof info.scce.dime.gui.gui.PrimitiveInputPort)) {
        return true;
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the primitive data type or null
   * 
   * @param node  A node from a data context of a GUI or Process model
   * @return data type or null
   */
  public PrimitiveType toPrimitve(final Node node) {
    PrimitiveType _switchResult = null;
    final Node it = node;
    boolean _matched = false;
    if (it instanceof PrimitiveAttribute) {
      _matched=true;
      _switchResult = ((PrimitiveAttribute)it).getAttribute().getDataType();
    }
    if (!_matched) {
      if (it instanceof PrimitiveListAttribute) {
        _matched=true;
        _switchResult = this._dataExtension.toData(((PrimitiveListAttribute)it).getPrimitiveType());
      }
    }
    if (!_matched) {
      if (it instanceof PrimitiveVariable) {
        _matched=true;
        _switchResult = this._dataExtension.toData(((PrimitiveVariable)it).getDataType());
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.gui.gui.PrimitiveInputPort) {
        _matched=true;
        _switchResult = this._dataExtension.toData(((info.scce.dime.gui.gui.PrimitiveInputPort)it).getDataType());
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.gui.gui.PrimitiveOutputPort) {
        _matched=true;
        _switchResult = this._dataExtension.toData(((info.scce.dime.gui.gui.PrimitiveOutputPort)it).getDataType());
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.process.process.PrimitiveAttribute) {
        _matched=true;
        _switchResult = ((info.scce.dime.process.process.PrimitiveAttribute)it).getAttribute().getDataType();
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.process.process.PrimitiveVariable) {
        _matched=true;
        _switchResult = this._dataExtension.toData(((info.scce.dime.process.process.PrimitiveVariable)it).getDataType());
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns the complex data type or null
   * 
   * @param node  A node from a data context of a GUI or Process model
   * @return data type or null
   */
  public Type toComplex(final Node it) {
    Type _switchResult = null;
    boolean _matched = false;
    if (it instanceof ComplexAttribute) {
      _matched=true;
      _switchResult = ((ComplexAttribute)it).getAttribute().getDataType();
    }
    if (!_matched) {
      if (it instanceof ComplexListAttribute) {
        _matched=true;
        _switchResult = ((ComplexListAttribute)it).getListType();
      }
    }
    if (!_matched) {
      if (it instanceof ComplexVariable) {
        _matched=true;
        _switchResult = ((ComplexVariable)it).getDataType();
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.gui.gui.ComplexInputPort) {
        _matched=true;
        _switchResult = ((info.scce.dime.gui.gui.ComplexInputPort)it).getDataType();
      }
    }
    if (!_matched) {
      if (it instanceof ComplexOutputPort) {
        _matched=true;
        _switchResult = ((ComplexOutputPort)it).getDataType();
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.process.process.ComplexAttribute) {
        _matched=true;
        _switchResult = ((info.scce.dime.process.process.ComplexAttribute)it).getAttribute().getDataType();
      }
    }
    if (!_matched) {
      if (it instanceof info.scce.dime.process.process.ComplexVariable) {
        _matched=true;
        _switchResult = ((info.scce.dime.process.process.ComplexVariable)it).getDataType();
      }
    }
    return _switchResult;
  }
  
  /**
   * Retrieves the data type that is referenced by the specified complex parameter
   * of a GUI plugin.
   * 
   * @param p  The complex parameter that references the data type to be retrieved.
   * @return  The Type that is referenced by the specified complex parameter, or
   *   {@code null} if not existent or the data model does not exist or cannot be
   *   accessed for whatever reason.
   */
  public Type toComplexType(final ComplexParameter p) {
    EObject _type = p.getType();
    return ((Type) _type);
  }
  
  /**
   * Retrieves the data type that is referenced by the specified complex parameter
   * of a GUI plugin.
   * 
   * @param p  The complex parameter that references the data type to be retrieved.
   * @return  The Type that is referenced by the specified complex parameter, or
   *   {@code null} if not existent or the data model does not exist or cannot be
   *   accessed for whatever reason.
   */
  public Type toComplexType(final ComplexInputParameter p) {
    return this.toComplexType(p.getParameter());
  }
  
  /**
   * Determines whether the specified button has the specified type.
   * 
   * @param button
   * @return {@code true} if this button is part of a {@code ButtonGroup} which
   *   itself is part of a {@code ButtonToolbar} that has the role {@code type}.
   */
  public boolean hasType(final Button button, final ButtonToolbarType type) {
    ButtonToolbar _findFirstParent = this.<ButtonToolbar>findFirstParent(button, ButtonToolbar.class);
    ButtonToolbarType _role = null;
    if (_findFirstParent!=null) {
      _role=_findFirstParent.getRole();
    }
    return Objects.equal(_role, type);
  }
  
  /**
   * Collects all input variables present in the given GUI model
   */
  public Iterable<Variable> inputVariables(final GUI gui) {
    final Function1<Variable, Boolean> _function = new Function1<Variable, Boolean>() {
      @Override
      public Boolean apply(final Variable it) {
        return Boolean.valueOf(it.isIsInput());
      }
    };
    return IterableExtensions.<Variable>filter(this.topLevelVariables(gui), _function);
  }
  
  /**
   * Collects all input variables present in the given GUI model
   */
  public Iterable<Variable> topLevelVariables(final GUI gui) {
    final Function1<DataContext, EList<Variable>> _function = new Function1<DataContext, EList<Variable>>() {
      @Override
      public EList<Variable> apply(final DataContext it) {
        return it.getVariables();
      }
    };
    final Function1<Variable, Boolean> _function_1 = new Function1<Variable, Boolean>() {
      @Override
      public Boolean apply(final Variable it) {
        return Boolean.valueOf((it.<ComplexAttributeConnector>getIncoming(ComplexAttributeConnector.class).isEmpty() && it.<ComplexListAttributeConnector>getIncoming(ComplexListAttributeConnector.class).isEmpty()));
      }
    };
    return IterableExtensions.<Variable>filter(Iterables.<Variable>concat(ListExtensions.<DataContext, EList<Variable>>map(gui.getDataContexts(), _function)), _function_1);
  }
  
  public List<GUIBranchPort> equallyNamedPorts(final List<GUIBranchPort> entry) {
    final LinkedList<GUIBranchPort> ports = new LinkedList<GUIBranchPort>();
    final LinkedList<Object> knownPorts = new LinkedList<Object>();
    final Consumer<GUIBranchPort> _function = new Consumer<GUIBranchPort>() {
      @Override
      public void accept(final GUIBranchPort n) {
        boolean _contains = knownPorts.contains(n.getName());
        if (_contains) {
          ports.add(n);
        } else {
          knownPorts.add(n.getName());
        }
        knownPorts.add(n);
      }
    };
    entry.forEach(_function);
    return ports;
  }
  
  public List<GUIBranchPort> notEquallyNamedPorts(final Iterable<GUIBranchPort> ports) {
    final Function1<GUIBranchPort, String> _function = new Function1<GUIBranchPort, String>() {
      @Override
      public String apply(final GUIBranchPort it) {
        return it.getName();
      }
    };
    final Function1<Map.Entry<String, List<GUIBranchPort>>, GUIBranchPort> _function_1 = new Function1<Map.Entry<String, List<GUIBranchPort>>, GUIBranchPort>() {
      @Override
      public GUIBranchPort apply(final Map.Entry<String, List<GUIBranchPort>> it) {
        return it.getValue().get(0);
      }
    };
    return IterableExtensions.<GUIBranchPort>toList(IterableExtensions.<Map.Entry<String, List<GUIBranchPort>>, GUIBranchPort>map(IterableExtensions.<String, GUIBranchPort>groupBy(ports, _function).entrySet(), _function_1));
  }
  
  public Collection<GUIBranch> getGUIBranchesMerged(final ModelElementContainer container) {
    final Function1<GUIBranch, String> _function = new Function1<GUIBranch, String>() {
      @Override
      public String apply(final GUIBranch it) {
        return it.getName();
      }
    };
    final Function1<List<GUIBranch>, GUIBranch> _function_1 = new Function1<List<GUIBranch>, GUIBranch>() {
      @Override
      public GUIBranch apply(final List<GUIBranch> it) {
        final Function2<GUIBranch, GUIBranch, GUIBranch> _function = new Function2<GUIBranch, GUIBranch, GUIBranch>() {
          @Override
          public GUIBranch apply(final GUIBranch x, final GUIBranch y) {
            GUIBranch _xifexpression = null;
            boolean _hasGUIOrigin = x.hasGUIOrigin();
            if (_hasGUIOrigin) {
              _xifexpression = x.mergedWith(y);
            } else {
              _xifexpression = y.mergedWith(x);
            }
            return _xifexpression;
          }
        };
        return IterableExtensions.<GUIBranch>reduce(it, _function);
      }
    };
    return MapExtensions.<String, List<GUIBranch>, GUIBranch>mapValues(IterableExtensions.<String, GUIBranch>groupBy(this.getGUIBranches(container, true), _function), _function_1).values();
  }
  
  /**
   * Collects model elements that imply a GUI branch.
   * These objects can be Buttons, GUIPlugin outputs or EndSIBs of embedded processes.
   * Searches deeply, i.e. recursively digs down into GUIs that are embedded via GUISIBs.
   */
  public List<GUIBranch> getGUIBranches(final ModelElementContainer container, final boolean merge) {
    List<GUIBranch> _xifexpression = null;
    if (this.cacheValues) {
      Collection<GUIBranch> branches = this._branchElements.get(container);
      if ((branches == null)) {
        branches = this.calcGUIBranches(container, CollectionLiterals.<ModelElementContainer>newHashSet(), GUIBranch.ALL_BRANCH_ELEMENT_TYPES, merge);
        this._branchElements.put(container, branches);
      }
      return IterableExtensions.<GUIBranch>toList(branches);
    } else {
      _xifexpression = IterableExtensions.<GUIBranch>toList(this.calcGUIBranches(container, CollectionLiterals.<ModelElementContainer>newHashSet(), GUIBranch.ALL_BRANCH_ELEMENT_TYPES, merge));
    }
    return _xifexpression;
  }
  
  public List<GUIBranch> getProcessBranches(final ModelElementContainer container) {
    List<GUIBranch> _xifexpression = null;
    if (this.cacheValues) {
      Collection<GUIBranch> branches = this._processBranchElements.get(container);
      if ((branches == null)) {
        branches = this.calcGUIBranches(container, CollectionLiterals.<ModelElementContainer>newHashSet(), GUIBranch.PROCESS_BRANCH_ELEMENT_TYPES, false);
        this._processBranchElements.put(container, branches);
      }
      return IterableExtensions.<GUIBranch>toList(branches);
    } else {
      _xifexpression = IterableExtensions.<GUIBranch>toList(this.calcGUIBranches(container, CollectionLiterals.<ModelElementContainer>newHashSet(), GUIBranch.PROCESS_BRANCH_ELEMENT_TYPES, false));
    }
    return _xifexpression;
  }
  
  private Collection<GUIBranch> calcGUIBranches(final ModelElementContainer container, final Set<ModelElementContainer> seen, final List<? extends Class<? extends ModelElement>> branchElements, final boolean merge) {
    List<GUIBranch> _xblockexpression = null;
    {
      boolean _add = seen.add(container);
      boolean _not = (!_add);
      if (_not) {
        return Collections.<GUIBranch>unmodifiableList(CollectionLiterals.<GUIBranch>newArrayList());
      }
      final Function1<ModelElement, Boolean> _function = new Function1<ModelElement, Boolean>() {
        @Override
        public Boolean apply(final ModelElement it) {
          boolean _switchResult = false;
          boolean _matched = false;
          if (it instanceof Button) {
            _matched=true;
            _switchResult = GUIBranchPort.isBranchable(((Button)it));
          }
          if (!_matched) {
            if (it instanceof GUISIB) {
              _matched=true;
              boolean _hasParent = GUIExtension.this.<ExtensionContext>hasParent(it, ExtensionContext.class);
              _switchResult = (!_hasParent);
            }
          }
          if (!_matched) {
            _switchResult = true;
          }
          return Boolean.valueOf(_switchResult);
        }
      };
      final Function1<ModelElement, Iterable<? extends IdentifiableElement>> _function_1 = new Function1<ModelElement, Iterable<? extends IdentifiableElement>>() {
        @Override
        public Iterable<? extends IdentifiableElement> apply(final ModelElement it) {
          Iterable<? extends IdentifiableElement> _switchResult = null;
          boolean _matched = false;
          if (it instanceof DispatchedGUISIB) {
            _matched=true;
            Set<GUI> _subGUIs = GUIExtension.this.getSubGUIs(((DispatchedGUISIB)it).getGui());
            GUI _gui = ((DispatchedGUISIB)it).getGui();
            _switchResult = Iterables.<GUI>concat(_subGUIs, Collections.<GUI>unmodifiableList(CollectionLiterals.<GUI>newArrayList(_gui)));
          }
          if (!_matched) {
            _switchResult = Collections.<ModelElement>unmodifiableList(CollectionLiterals.<ModelElement>newArrayList(it));
          }
          return _switchResult;
        }
      };
      final Function1<IdentifiableElement, Iterable<GUIBranch>> _function_2 = new Function1<IdentifiableElement, Iterable<GUIBranch>>() {
        @Override
        public Iterable<GUIBranch> apply(final IdentifiableElement it) {
          Iterable<GUIBranch> _switchResult = null;
          boolean _matched = false;
          if (it instanceof GUI) {
            _matched=true;
            _switchResult = GUIExtension.this.calcGUIBranches(((ModelElementContainer)it), seen, branchElements, merge);
          }
          if (!_matched) {
            if (it instanceof GUISIB) {
              _matched=true;
              _switchResult = GUIExtension.this.<ModelElement>findSelectiveNodes(((GUISIB)it), seen, branchElements, merge);
            }
          }
          if (!_matched) {
            if (it instanceof GUIPlugin) {
              _matched=true;
              final Function1<Branch, GUIBranch> _function = new Function1<Branch, GUIBranch>() {
                @Override
                public GUIBranch apply(final Branch it) {
                  return GUIBranch.toGUIBranch(it);
                }
              };
              _switchResult = IterableExtensions.<Branch, GUIBranch>map(GUIExtension.this.pluginOutputs(((GUIPlugin)it)), _function);
            }
          }
          if (!_matched) {
            if (it instanceof ProcessSIB) {
              _matched=true;
              final Function1<EndSIB, GUIBranch> _function = new Function1<EndSIB, GUIBranch>() {
                @Override
                public GUIBranch apply(final EndSIB it) {
                  return GUIBranch.toGUIBranch(it);
                }
              };
              _switchResult = IterableExtensions.<EndSIB, GUIBranch>map(GUIExtension.this.getProcessEndSIBs(((ProcessSIB)it)), _function);
            }
          }
          if (!_matched) {
            if (it instanceof ModelElement) {
              _matched=true;
              GUIBranch _gUIBranch = GUIBranch.toGUIBranch(((ModelElement)it));
              _switchResult = Collections.<GUIBranch>unmodifiableList(CollectionLiterals.<GUIBranch>newArrayList(_gUIBranch));
            }
          }
          if (!_matched) {
            _switchResult = Collections.<GUIBranch>unmodifiableList(CollectionLiterals.<GUIBranch>newArrayList());
          }
          return _switchResult;
        }
      };
      final Iterable<GUIBranch> res = IterableExtensions.<IdentifiableElement, GUIBranch>flatMap(Iterables.<IdentifiableElement>concat(IterableExtensions.<ModelElement, Iterable<? extends IdentifiableElement>>map(IterableExtensions.<ModelElement>filter(Iterables.<ModelElement>filter(this.findDeeply(container, ((Class<? extends ModelElement>[])Conversions.unwrapArray(branchElements, Class.class)), this.digIntoNonExtendingGUISIBs), ModelElement.class), _function), _function_1)), _function_2);
      if (merge) {
        return GUIBranch.merge(res);
      }
      _xblockexpression = IterableExtensions.<GUIBranch>toList(res);
    }
    return _xblockexpression;
  }
  
  private <C extends ModelElement> List<GUIBranch> dispatchBranches(final GUI parentGUI, final Iterable<GUI> subGUIs, final Set<ModelElementContainer> seen, final List<? extends Class<? extends ModelElement>> branchElements, final boolean merge) {
    boolean _contains = seen.contains(parentGUI);
    if (_contains) {
      return Collections.<GUIBranch>unmodifiableList(CollectionLiterals.<GUIBranch>newArrayList());
    }
    final Function1<GUIBranch, String> _function = new Function1<GUIBranch, String>() {
      @Override
      public String apply(final GUIBranch it) {
        return it.getName();
      }
    };
    final Map<String, GUIBranch> branchesByName = IterableExtensions.<String, GUIBranch>toMap(this.calcGUIBranches(parentGUI, seen, branchElements, merge), _function);
    seen.add(parentGUI);
    final Function1<GUI, Boolean> _function_1 = new Function1<GUI, Boolean>() {
      @Override
      public Boolean apply(final GUI it) {
        boolean _contains = seen.contains(it);
        return Boolean.valueOf((!_contains));
      }
    };
    final Function1<GUI, Collection<GUIBranch>> _function_2 = new Function1<GUI, Collection<GUIBranch>>() {
      @Override
      public Collection<GUIBranch> apply(final GUI it) {
        return GUIExtension.this.calcGUIBranches(it, seen, branchElements, merge);
      }
    };
    final Consumer<GUIBranch> _function_3 = new Consumer<GUIBranch>() {
      @Override
      public void accept(final GUIBranch subGUIBranch) {
        final GUIBranch knownBranch = branchesByName.get(subGUIBranch.getName());
        GUIBranch _elvis = null;
        GUIBranch _mergedWith = null;
        if (knownBranch!=null) {
          _mergedWith=knownBranch.mergedWith(subGUIBranch);
        }
        if (_mergedWith != null) {
          _elvis = _mergedWith;
        } else {
          _elvis = subGUIBranch;
        }
        branchesByName.put(
          subGUIBranch.getName(), _elvis);
      }
    };
    IterableExtensions.<GUI, GUIBranch>flatMap(IterableExtensions.<GUI>filter(subGUIs, _function_1), _function_2).forEach(_function_3);
    return IterableExtensions.<GUIBranch>toList(branchesByName.values());
  }
  
  private <C extends ModelElement> Collection<GUIBranch> findSelectiveNodes(final GUISIB sib, final Set<ModelElementContainer> seen, final List<? extends Class<? extends ModelElement>> branchElements, final boolean merge) {
    final Collection<GUIBranch> guiSIBBranches = this.calcGUIBranches(sib.getGui(), seen, branchElements, merge);
    GUISIBModal _modal = sib.getModal();
    boolean _tripleNotEquals = (_modal != null);
    if (_tripleNotEquals) {
      final Function1<GUIBranch, Boolean> _function = new Function1<GUIBranch, Boolean>() {
        @Override
        public Boolean apply(final GUIBranch it) {
          final Function1<ModalCloseButton, String> _function = new Function1<ModalCloseButton, String>() {
            @Override
            public String apply(final ModalCloseButton it) {
              return it.getBranchName();
            }
          };
          boolean _contains = ListExtensions.<ModalCloseButton, String>map(sib.getModal().getModalCloseButton(), _function).contains(it.getName());
          return Boolean.valueOf((!_contains));
        }
      };
      final List<GUIBranch> bs = IterableExtensions.<GUIBranch>toList(IterableExtensions.<GUIBranch>filter(guiSIBBranches, _function));
      return bs;
    }
    return guiSIBBranches;
  }
  
  public boolean modalCloseButton(final GUISIB guisib, final String branchName) {
    boolean _xblockexpression = false;
    {
      GUISIBModal _modal = guisib.getModal();
      boolean _tripleNotEquals = (_modal != null);
      if (_tripleNotEquals) {
        final Function1<ModalCloseButton, String> _function = new Function1<ModalCloseButton, String>() {
          @Override
          public String apply(final ModalCloseButton it) {
            return branchName;
          }
        };
        return ListExtensions.<ModalCloseButton, String>map(guisib.getModal().getModalCloseButton(), _function).contains(branchName);
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  public Iterable<EndSIB> getProcessEndSIBs(final ProcessSIB ip) {
    info.scce.dime.process.process.Process _proMod = ip.getProMod();
    return ((info.scce.dime.process.process.Process) _proMod).getEndSIBs();
  }
  
  /**
   * Collects all events present in the given GUI model listener contexts
   */
  public Iterable<Event> events(final GUI gui) {
    final Function1<ListenerContext, EList<Event>> _function = new Function1<ListenerContext, EList<Event>>() {
      @Override
      public EList<Event> apply(final ListenerContext it) {
        return it.getEvents();
      }
    };
    return Iterables.<Event>concat(ListExtensions.<ListenerContext, EList<Event>>map(gui.getListenerContexts(), _function));
  }
  
  /**
   * Collects all outputs of a given GUI plug in component.
   * The outputs correspond to a branch of all GUI plug in SIBs
   * @param p the GUI plug in
   * @return List<Output> the list of outputs correspond to branches
   */
  public Iterable<Branch> pluginOutputs(final GUIPlugin p) {
    return Iterables.<Branch>filter(p.getAbstractBranchs(), Branch.class);
  }
  
  /**
   * Retrieves all sub-GUIs of the specified GUI.
   * 
   * @param gui - The GUI for which to retrieve sub-GUIs.
   * @return The list of sub-GUIs.
   */
  public Set<GUI> getSubGUIs(final GUI gui) {
    return this.getSubGUIsMap().get(gui);
  }
  
  private Map<GUI, Set<GUI>> getSubGUIsMap() {
    Map<GUI, Set<GUI>> _xifexpression = null;
    if (this.cacheValues) {
      Map<GUI, Set<GUI>> _elvis = null;
      if (this._subGUIs != null) {
        _elvis = this._subGUIs;
      } else {
        _elvis = (this._subGUIs = this.calcSubGUIsMap());
      }
      _xifexpression = _elvis;
    } else {
      _xifexpression = this.calcSubGUIsMap();
    }
    return _xifexpression;
  }
  
  private HashMap<GUI, Set<GUI>> calcSubGUIsMap() {
    HashMap<GUI, Set<GUI>> _newHashMap = CollectionLiterals.<GUI, Set<GUI>>newHashMap();
    final Procedure1<HashMap<GUI, Set<GUI>>> _function = new Procedure1<HashMap<GUI, Set<GUI>>>() {
      @Override
      public void apply(final HashMap<GUI, Set<GUI>> it) {
        final Consumer<GUI> _function = new Consumer<GUI>() {
          @Override
          public void accept(final GUI gui) {
            it.put(gui, GUIExtension.this.calcExtendingGUIs(gui));
          }
        };
        ReferenceRegistry.getInstance().<GUI>lookup(GUI.class).forEach(_function);
      }
    };
    return ObjectExtensions.<HashMap<GUI, Set<GUI>>>operator_doubleArrow(_newHashMap, _function);
  }
  
  private Set<GUI> calcExtendingGUIs(final GUI gui) {
    final Function1<List<GUI>, Boolean> _function = new Function1<List<GUI>, Boolean>() {
      @Override
      public Boolean apply(final List<GUI> it) {
        return Boolean.valueOf(it.contains(gui));
      }
    };
    final Function1<List<GUI>, List<GUI>> _function_1 = new Function1<List<GUI>, List<GUI>>() {
      @Override
      public List<GUI> apply(final List<GUI> it) {
        return it.subList(0, it.indexOf(gui));
      }
    };
    return IterableExtensions.<GUI>toSet(IterableExtensions.<List<GUI>, GUI>flatMap(IterableExtensions.<List<GUI>>filter(this.getSuperGUIsMap().values(), _function), _function_1));
  }
  
  /**
   * Collects all direct sub GUI models for which the given GUI model is the parent
   * @param gui the parent GUI model
   * @param allGUIs list of all GUI models which should be considered as sub GUI model
   * @return Iterable<GUI> list of all direct sub GUI models
   */
  public List<GUI> findAllSubGUIs(final GUI gui, final Map<Integer, List<GUI>> sortedSubGuis) {
    final Function1<Map.Entry<Integer, List<GUI>>, Boolean> _function = new Function1<Map.Entry<Integer, List<GUI>>, Boolean>() {
      @Override
      public Boolean apply(final Map.Entry<Integer, List<GUI>> it) {
        final Function1<GUI, Boolean> _function = new Function1<GUI, Boolean>() {
          @Override
          public Boolean apply(final GUI n) {
            return Boolean.valueOf(n.getId().equals(gui.getId()));
          }
        };
        GUI _findFirst = IterableExtensions.<GUI>findFirst(it.getValue(), _function);
        return Boolean.valueOf(GUIExtension.this.operator_notEquals(_findFirst, null));
      }
    };
    final Integer key = IterableExtensions.<Map.Entry<Integer, List<GUI>>>findFirst(sortedSubGuis.entrySet(), _function).getKey();
    final LinkedList<GUI> list = new LinkedList<GUI>();
    int i = ((key).intValue() - 1);
    while ((i >= 0)) {
      {
        List<GUI> _get = sortedSubGuis.get(Integer.valueOf(i));
        Iterables.<GUI>addAll(list, _get);
        i--;
      }
    }
    return list;
  }
  
  /**
   * Collects all sub GUI models for which the given GUI model is the parent
   * @param gui the parent GUI model
   * @param allGUIs list of all GUI models which should be considered as sub GUI model
   * @return Iterable<GUI> list of all direct sub GUI models
   */
  public List<GUI> findAllSubGUIs(final GUISIB guiSib, final Map<Integer, List<GUI>> allGUIs) {
    return this.findAllSubGUIs(guiSib.getGui(), allGUIs);
  }
  
  /**
   * Calculates the data model type of the given complex input port
   * which is specialized by the equally named input port of the given sub GUI model
   * @param port the complex input port of a dispatched GUI SIB
   * @param subGUI one of the specialized sub GUI models which extends the type of the given port
   * @return Type the type of the port or a specialized sub Type
   */
  public Type toSpecialType(final info.scce.dime.gui.gui.ComplexInputPort port, final GUI subGUI) {
    final Function1<ComplexVariable, Boolean> _function = new Function1<ComplexVariable, Boolean>() {
      @Override
      public Boolean apply(final ComplexVariable it) {
        return Boolean.valueOf(it.getName().equals(port.getName()));
      }
    };
    return (((ComplexVariable[])Conversions.unwrapArray(IterableExtensions.<ComplexVariable>filter(Iterables.<ComplexVariable>filter(this.inputVariables(subGUI), ComplexVariable.class), _function), ComplexVariable.class))[0]).getDataType();
  }
  
  public List<GUI> getSuperGUIs(final GUI gui) {
    return this.getSuperGUIsMap().get(gui);
  }
  
  private Map<GUI, List<GUI>> getSuperGUIsMap() {
    Map<GUI, List<GUI>> _xifexpression = null;
    if (this.cacheValues) {
      Map<GUI, List<GUI>> _elvis = null;
      if (this._superGUIs != null) {
        _elvis = this._superGUIs;
      } else {
        _elvis = (this._superGUIs = this.calcSuperGUIsMap());
      }
      _xifexpression = _elvis;
    } else {
      _xifexpression = this.calcSuperGUIsMap();
    }
    return _xifexpression;
  }
  
  private HashMap<GUI, List<GUI>> calcSuperGUIsMap() {
    HashMap<GUI, List<GUI>> _newHashMap = CollectionLiterals.<GUI, List<GUI>>newHashMap();
    final Procedure1<HashMap<GUI, List<GUI>>> _function = new Procedure1<HashMap<GUI, List<GUI>>>() {
      @Override
      public void apply(final HashMap<GUI, List<GUI>> it) {
        final Consumer<GUI> _function = new Consumer<GUI>() {
          @Override
          public void accept(final GUI gui) {
            it.put(gui, GUIExtension.this.calcExtensionHierarchy(gui));
          }
        };
        ReferenceRegistry.getInstance().<GUI>lookup(GUI.class).forEach(_function);
      }
    };
    return ObjectExtensions.<HashMap<GUI, List<GUI>>>operator_doubleArrow(_newHashMap, _function);
  }
  
  private List<GUI> calcExtensionHierarchy(final GUI gui) {
    List<GUI> _elvis = null;
    GUI _extendedGUI = this.getExtendedGUI(gui);
    List<GUI> _calcExtensionHierarchy = null;
    if (_extendedGUI!=null) {
      _calcExtensionHierarchy=this.calcExtensionHierarchy(_extendedGUI);
    }
    if (_calcExtensionHierarchy != null) {
      _elvis = _calcExtensionHierarchy;
    } else {
      _elvis = Collections.<GUI>unmodifiableList(CollectionLiterals.<GUI>newArrayList());
    }
    return IterableExtensions.<GUI>toList(Iterables.<GUI>concat(Collections.<GUI>unmodifiableList(CollectionLiterals.<GUI>newArrayList(gui)), _elvis));
  }
  
  public GUI getExtendedGUI(final GUI it) {
    return IterableExtensions.<GUI>head(this.getExtendedGUIs(it));
  }
  
  public Iterable<GUI> getExtendedGUIs(final GUI it) {
    final Function1<ExtensionContext, List<GUISIB>> _function = new Function1<ExtensionContext, List<GUISIB>>() {
      @Override
      public List<GUISIB> apply(final ExtensionContext it) {
        return GUIExtension.this.<GUISIB>find(it, GUISIB.class);
      }
    };
    final Function1<GUISIB, GUI> _function_1 = new Function1<GUISIB, GUI>() {
      @Override
      public GUI apply(final GUISIB it) {
        return it.getGui();
      }
    };
    return IterableExtensions.<GUISIB, GUI>map(IterableExtensions.<ExtensionContext, GUISIB>flatMap(it.getExtensionContexts(), _function), _function_1);
  }
  
  public List<GUIBranch> dispatchedOutputs(final GUI gui) {
    return this.<ModelElement>dispatchBranches(gui, this.getSubGUIs(gui), CollectionLiterals.<ModelElementContainer>newHashSet(), GUIBranch.ALL_BRANCH_ELEMENT_TYPES, true);
  }
  
  public List<GUIBranch> dispatchedOutputs(final GUI parentGUI, final GUI dispatchedGUI) {
    return this.<ModelElement>dispatchBranches(parentGUI, Collections.<GUI>unmodifiableList(CollectionLiterals.<GUI>newArrayList(dispatchedGUI)), CollectionLiterals.<ModelElementContainer>newHashSet(), GUIBranch.ALL_BRANCH_ELEMENT_TYPES, true);
  }
  
  public Map<Integer, List<GUI>> getSortedDispatchedGUIs(final GUI parentGUI) {
    return this.getSortedDispatchedGUIs(parentGUI, this.getSubGUIs(parentGUI));
  }
  
  public Map<Integer, List<GUI>> getSortedDispatchedGUIs(final GUI parentGUI, final Collection<GUI> specializedGUI) {
    final LinkedList<GUI> spezAndSelf = new LinkedList<GUI>();
    spezAndSelf.addAll(specializedGUI);
    spezAndSelf.add(parentGUI);
    final Function1<GUI, Integer> _function = new Function1<GUI, Integer>() {
      @Override
      public Integer apply(final GUI it) {
        return Integer.valueOf(GUIExtension.this.inputScore(it, GUIExtension.this.inputVariables(parentGUI)));
      }
    };
    final Map<Integer, List<GUI>> scoredGUIList = IterableExtensions.<Integer, GUI>groupBy(spezAndSelf, _function);
    final Function1<Map.Entry<Integer, List<GUI>>, Integer> _function_1 = new Function1<Map.Entry<Integer, List<GUI>>, Integer>() {
      @Override
      public Integer apply(final Map.Entry<Integer, List<GUI>> it) {
        return it.getKey();
      }
    };
    final List<Map.Entry<Integer, List<GUI>>> sortedByInputScore = ListExtensions.<Map.Entry<Integer, List<GUI>>>reverse(IterableExtensions.<Map.Entry<Integer, List<GUI>>, Integer>sortBy(scoredGUIList.entrySet(), _function_1));
    final Map<Integer, List<GUI>> resultMap = new HashMap<Integer, List<GUI>>();
    int i = 0;
    for (final Map.Entry<Integer, List<GUI>> s : sortedByInputScore) {
      {
        final Function1<GUI, Integer> _function_2 = new Function1<GUI, Integer>() {
          @Override
          public Integer apply(final GUI it) {
            return Integer.valueOf(GUIExtension.this.hierachyScore(it));
          }
        };
        final Function1<Map.Entry<Integer, List<GUI>>, Integer> _function_3 = new Function1<Map.Entry<Integer, List<GUI>>, Integer>() {
          @Override
          public Integer apply(final Map.Entry<Integer, List<GUI>> it) {
            return it.getKey();
          }
        };
        final List<Map.Entry<Integer, List<GUI>>> sortedByHierachyScore = ListExtensions.<Map.Entry<Integer, List<GUI>>>reverse(IterableExtensions.<Map.Entry<Integer, List<GUI>>, Integer>sortBy(IterableExtensions.<Integer, GUI>groupBy(s.getValue(), _function_2).entrySet(), _function_3));
        for (final Map.Entry<Integer, List<GUI>> h : sortedByHierachyScore) {
          {
            boolean _containsKey = resultMap.containsKey(Integer.valueOf(i));
            boolean _not = (!_containsKey);
            if (_not) {
              LinkedList<GUI> _linkedList = new LinkedList<GUI>();
              resultMap.put(Integer.valueOf(i), _linkedList);
            }
            resultMap.get(Integer.valueOf(i)).addAll(h.getValue());
            i++;
          }
        }
      }
    }
    return resultMap;
  }
  
  private int inputScore(final GUI g, final Iterable<Variable> parentInputs) {
    final Function1<ComplexVariable, Boolean> _function = new Function1<ComplexVariable, Boolean>() {
      @Override
      public Boolean apply(final ComplexVariable v) {
        final Function1<Variable, Boolean> _function = new Function1<Variable, Boolean>() {
          @Override
          public Boolean apply(final Variable it) {
            return Boolean.valueOf(it.getName().equals(v.getName()));
          }
        };
        Variable _findFirst = IterableExtensions.<Variable>findFirst(parentInputs, _function);
        return Boolean.valueOf(GUIExtension.this.operator_notEquals(_findFirst, null));
      }
    };
    final Function1<ComplexVariable, Integer> _function_1 = new Function1<ComplexVariable, Integer>() {
      @Override
      public Integer apply(final ComplexVariable it) {
        return Integer.valueOf(GUIExtension.this.getSingleScore(it));
      }
    };
    final Function2<Integer, Integer, Integer> _function_2 = new Function2<Integer, Integer, Integer>() {
      @Override
      public Integer apply(final Integer sum, final Integer score) {
        return Integer.valueOf(((score).intValue() + (sum).intValue()));
      }
    };
    final Integer sum = IterableExtensions.<Integer>reduce(IterableExtensions.<ComplexVariable, Integer>map(IterableExtensions.<ComplexVariable>filter(Iterables.<ComplexVariable>filter(this.inputVariables(g), ComplexVariable.class), _function), _function_1), _function_2);
    if ((sum == null)) {
      return 0;
    }
    return (sum).intValue();
  }
  
  private int hierachyScore(final GUI gui) {
    return this.getSuperGUIs(gui).size();
  }
  
  private int getSingleScore(final ComplexVariable v) {
    return IterableExtensions.size(this._dataExtension.getSuperTypes(v.getDataType()));
  }
  
  /**
   * Retrieves the top-level variable of an attribute in the data context.
   */
  public Variable getTopLevelVariable(final Attribute it) {
    Variable _rootVariable = this.getRootVariable(it);
    Variable _topLevelVariable = null;
    if (_rootVariable!=null) {
      _topLevelVariable=this.getTopLevelVariable(_rootVariable);
    }
    return _topLevelVariable;
  }
  
  /**
   * Retrieves the top-level variable of an attribute if it is represented
   * as a node of type {@link Variable} that is connected to the actual
   * variable via an {@link AttributeConnector}.
   */
  public Variable getTopLevelVariable(final Variable it) {
    Variable _elvis = null;
    Variable _rootVariable = this.getRootVariable(it);
    Variable _topLevelVariable = null;
    if (_rootVariable!=null) {
      _topLevelVariable=this.getTopLevelVariable(_rootVariable);
    }
    if (_topLevelVariable != null) {
      _elvis = _topLevelVariable;
    } else {
      _elvis = it;
    }
    return _elvis;
  }
  
  /**
   * Retrieves the root variable of an attribute.
   */
  public Variable getRootVariable(final Attribute attribute) {
    ModelElementContainer _container = attribute.getContainer();
    return ((Variable) _container);
  }
  
  /**
   * Retrieves the root variable of an attribute if it is represented
   * as a node of type {@link Variable} that is connected to the actual
   * variable via an {@link AttributeConnector}.
   */
  public Variable getRootVariable(final Variable attribute) {
    Node _elvis = null;
    Node _findSourceOf = this.<ComplexAttributeConnector>findSourceOf(attribute, ComplexAttributeConnector.class);
    if (_findSourceOf != null) {
      _elvis = _findSourceOf;
    } else {
      Node _findSourceOf_1 = this.<ComplexListAttributeConnector>findSourceOf(attribute, ComplexListAttributeConnector.class);
      _elvis = _findSourceOf_1;
    }
    return ((Variable) _elvis);
  }
  
  /**
   * Retrieves the name of an attribute.
   */
  public String getName(final Attribute it) {
    String _switchResult = null;
    boolean _matched = false;
    if (it instanceof PrimitiveAttribute) {
      _matched=true;
      _switchResult = ((PrimitiveAttribute)it).getAttribute().getName();
    }
    if (!_matched) {
      if (it instanceof PrimitiveListAttribute) {
        _matched=true;
        _switchResult = ((PrimitiveListAttribute)it).getAttributeName().toString();
      }
    }
    if (!_matched) {
      if (it instanceof ComplexAttribute) {
        _matched=true;
        _switchResult = ((ComplexAttribute)it).getAttribute().getName();
      }
    }
    if (!_matched) {
      if (it instanceof ComplexListAttribute) {
        _matched=true;
        _switchResult = ((ComplexListAttribute)it).getAttributeName().toString();
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns all top level variables of a given GUI model
   * @param gui
   * @return
   */
  public Iterable<Variable> findVariables(final GUI gui) {
    final Function1<DataContext, EList<Variable>> _function = new Function1<DataContext, EList<Variable>>() {
      @Override
      public EList<Variable> apply(final DataContext it) {
        return it.getVariables();
      }
    };
    final Function1<Variable, Boolean> _function_1 = new Function1<Variable, Boolean>() {
      @Override
      public Boolean apply(final Variable it) {
        return Boolean.valueOf(it.<ComplexAttributeConnector>getIncoming(ComplexAttributeConnector.class).isEmpty());
      }
    };
    final Function1<Variable, Boolean> _function_2 = new Function1<Variable, Boolean>() {
      @Override
      public Boolean apply(final Variable it) {
        return Boolean.valueOf(it.<ComplexListAttributeConnector>getIncoming(ComplexListAttributeConnector.class).isEmpty());
      }
    };
    return IterableExtensions.<Variable>filter(IterableExtensions.<Variable>filter(Iterables.<Variable>concat(ListExtensions.<DataContext, EList<Variable>>map(gui.getDataContexts(), _function)), _function_1), _function_2);
  }
  
  /**
   * Finds all embedded processes inside the specified container.
   * Recurses into all sub-containers.
   * 
   * @param container - The container holding the elements to be searched through.
   * @return A set of processes. Might be empty but is never null.
   */
  public Set<info.scce.dime.process.process.Process> getEmbeddedProcesses(final ModelElementContainer container) {
    Set<info.scce.dime.process.process.Process> _xifexpression = null;
    if (this.cacheValues) {
      Set<info.scce.dime.process.process.Process> _elvis = null;
      Set<info.scce.dime.process.process.Process> _get = this._embeddedProcesses.get(container);
      if (_get != null) {
        _elvis = _get;
      } else {
        final Function1<ProcessSIB, info.scce.dime.process.process.Process> _function = new Function1<ProcessSIB, info.scce.dime.process.process.Process>() {
          @Override
          public info.scce.dime.process.process.Process apply(final ProcessSIB it) {
            info.scce.dime.process.process.Process _proMod = it.getProMod();
            return ((info.scce.dime.process.process.Process) _proMod);
          }
        };
        Set<info.scce.dime.process.process.Process> _set = IterableExtensions.<info.scce.dime.process.process.Process>toSet(ListExtensions.<ProcessSIB, info.scce.dime.process.process.Process>map(this.<ProcessSIB>find(container, ProcessSIB.class), _function));
        _elvis = _set;
      }
      final Procedure1<Set<info.scce.dime.process.process.Process>> _function_1 = new Procedure1<Set<info.scce.dime.process.process.Process>>() {
        @Override
        public void apply(final Set<info.scce.dime.process.process.Process> it) {
          GUIExtension.this._embeddedProcesses.put(container, it);
        }
      };
      _xifexpression = ObjectExtensions.<Set<info.scce.dime.process.process.Process>>operator_doubleArrow(_elvis, _function_1);
    } else {
      final Function1<ProcessSIB, info.scce.dime.process.process.Process> _function_2 = new Function1<ProcessSIB, info.scce.dime.process.process.Process>() {
        @Override
        public info.scce.dime.process.process.Process apply(final ProcessSIB it) {
          info.scce.dime.process.process.Process _proMod = it.getProMod();
          return ((info.scce.dime.process.process.Process) _proMod);
        }
      };
      _xifexpression = IterableExtensions.<info.scce.dime.process.process.Process>toSet(ListExtensions.<ProcessSIB, info.scce.dime.process.process.Process>map(this.<ProcessSIB>find(container, ProcessSIB.class), _function_2));
    }
    return _xifexpression;
  }
  
  /**
   * Finds the model element that is used in an iteration, i.e. it has an incoming
   * Iteration or TableLoad edge. This element is either the specified model element
   * itself or the first parent (bottom-up) that fulfills the condition.
   * 
   * @param elm - The element for which to find the loop container.
   * @return  The first matching parent, or {@code null} if none exists.
   */
  public Node getSurroundingLoop(final Node elm) {
    Node _xifexpression = null;
    if (this.cacheValues) {
      Node _elvis = null;
      Node _get = this._surroundingLoops.get(elm);
      if (_get != null) {
        _elvis = _get;
      } else {
        Node _calcSurroundingLoop = this.calcSurroundingLoop(elm);
        _elvis = _calcSurroundingLoop;
      }
      final Procedure1<Node> _function = new Procedure1<Node>() {
        @Override
        public void apply(final Node it) {
          GUIExtension.this._surroundingLoops.put(elm, it);
        }
      };
      _xifexpression = ObjectExtensions.<Node>operator_doubleArrow(_elvis, _function);
    } else {
      _xifexpression = this.calcSurroundingLoop(elm);
    }
    return _xifexpression;
  }
  
  public Node calcSurroundingLoop(final Node elm) {
    Node _xifexpression = null;
    boolean _isEmpty = this.findPredecessorsVia(elm, Iteration.class, TableLoad.class).isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      _xifexpression = elm;
    } else {
      final Function1<Container, Boolean> _function = new Function1<Container, Boolean>() {
        @Override
        public Boolean apply(final Container it) {
          boolean _isEmpty = GUIExtension.this.findPredecessorsVia(it, Iteration.class, TableLoad.class).isEmpty();
          return Boolean.valueOf((!_isEmpty));
        }
      };
      _xifexpression = this.<Container>findFirstParent(elm, Container.class, _function);
    }
    return _xifexpression;
  }
  
  /**
   * Finds all input elements that are relevant for the specified form only,
   * i.e. the search recurses into all sub-containers except Forms.
   * 
   * @param form - The Form container holding the elements to be searched through.
   * @return  A list of Inputs. Might be empty but is never null.
   */
  public List<Input> getFormInputs(final Form form) {
    List<Input> _xifexpression = null;
    if (this.cacheValues) {
      List<Input> _elvis = null;
      List<Input> _get = this._formInputs.get(form);
      if (_get != null) {
        _elvis = _get;
      } else {
        List<Input> _calcFormInputs = this.calcFormInputs(form);
        _elvis = _calcFormInputs;
      }
      final Procedure1<List<Input>> _function = new Procedure1<List<Input>>() {
        @Override
        public void apply(final List<Input> it) {
          GUIExtension.this._formInputs.put(form, it);
        }
      };
      _xifexpression = ObjectExtensions.<List<Input>>operator_doubleArrow(_elvis, _function);
    } else {
      _xifexpression = this.calcFormInputs(form);
    }
    return _xifexpression;
  }
  
  private List<Input> calcFormInputs(final Form form) {
    final Function1<Input, Boolean> _function = new Function1<Input, Boolean>() {
      @Override
      public Boolean apply(final Input it) {
        Form _findFirstParent = GUIExtension.this.<Form>findFirstParent(it, Form.class);
        return Boolean.valueOf(GUIExtension.this.operator_equals(_findFirstParent, form));
      }
    };
    return IterableExtensions.<Input>toList(IterableExtensions.<Input>filter(this.<Input>find(form, Input.class), _function));
  }
  
  /**
   * Finds all elements of specific type inside the specified container.
   * Recurses into all sub-containers.
   * 
   * @param container - The container holding the elements to be searched through.
   * @param clazz - The class of the elements to be found.
   * @return  A list of elements of the specified type. Might be empty but is never null.
   */
  @Override
  public <C extends IdentifiableElement> List<C> find(final ModelElementContainer container, final Class<C> clazz) {
    List<C> _xifexpression = null;
    if (this.cacheValues) {
      List<C> _elvis = null;
      List<? extends IdentifiableElement> _get = this._modelElements.get(container).get(clazz);
      if (((List<C>) _get) != null) {
        _elvis = ((List<C>) _get);
      } else {
        List<C> _list = IterableExtensions.<C>toList(super.<C>find(container, clazz));
        final Procedure1<List<C>> _function = new Procedure1<List<C>>() {
          @Override
          public void apply(final List<C> it) {
            GUIExtension.this._modelElements.get(container).put(clazz, it);
          }
        };
        List<C> _doubleArrow = ObjectExtensions.<List<C>>operator_doubleArrow(_list, _function);
        _elvis = _doubleArrow;
      }
      _xifexpression = _elvis;
    } else {
      _xifexpression = IterableExtensions.<C>toList(super.<C>find(container, clazz));
    }
    return _xifexpression;
  }
  
  /**
   * Finds the first parent (bottom-up) of a model element of a specific type.
   * 
   * @param elm - The element for which to find the parent elements.
   * @param cls - The class of the parent elements to be found.
   * @return  The first parent of the specified type, or {@code null} if none exists.
   */
  @Override
  public <E extends ModelElement> E findFirstParent(final ModelElement elm, final Class<E> cls) {
    E _xifexpression = null;
    if (this.cacheValues) {
      E _elvis = null;
      ModelElement _get = this._firstParents.get(elm).get(cls);
      if (((E) _get) != null) {
        _elvis = ((E) _get);
      } else {
        E _findFirstParent = super.<E>findFirstParent(elm, cls);
        final Procedure1<E> _function = new Procedure1<E>() {
          @Override
          public void apply(final E it) {
            GUIExtension.this._firstParents.get(elm).put(cls, it);
          }
        };
        E _doubleArrow = ObjectExtensions.<E>operator_doubleArrow(_findFirstParent, _function);
        _elvis = _doubleArrow;
      }
      _xifexpression = _elvis;
    } else {
      _xifexpression = super.<E>findFirstParent(elm, cls);
    }
    return _xifexpression;
  }
  
  /**
   * Find all elements of specific type inside the specified container.
   * <br>Recurses into all sub-containers.
   * <br>Recurses into GUI sub-models that are integrated via GUISIBs.
   * <br>The type of each element is matched to the specified type via {@code instanceof}.
   * 
   * <p>Convenient method for {@code findDeeply(container, cls, [switch it { GUISIB: it.gui }])}</p>
   * 
   * @param container  The container holding the elements to be searched through.
   * @param cls  The class of the elements to be found.
   * @return  A set of elements of the specified type. Might be empty but is never {@code null}.
   */
  public <C extends ModelElement> Iterable<C> findDeepInGUIs(final ModelElementContainer it, final Class<C> cls) {
    return this.<C>findDeeply(it, cls, this.digIntoGUISIBs);
  }
  
  /**
   * Find all elements of any of the specified types inside the specified container.
   * <br>Recurses into all sub-containers.
   * <br>Recurses into GUI sub-models that are integrated via GUISIBs.
   * <br>The type of each element is matched to the specified types via {@code instanceof}.
   * 
   * <p>Convenient method for {@code findDeeply(container, classes, [switch it { GUISIB: it.gui }])}</p>
   * 
   * @param container  The container holding the elements to be searched through.
   * @param classes  A list of classes. The elements to be found must be instance of at least one of them.
   * @return  A set of elements of any of the specified types. Might be empty but never {@code null}.
   */
  public <C extends ModelElement> Iterable<C> findDeepInGUIs(final ModelElementContainer it, final Class<C>[] classes) {
    return this.<C>findDeeply(it, classes, this.digIntoGUISIBs);
  }
  
  public String placeholderTransclusion(final String name, final String id) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("select=\"[dime");
    {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(id);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("-");
      }
    }
    _builder.append(id);
    _builder.append("-placeholder=");
    _builder.append(name);
    _builder.append("]\"");
    return _builder.toString();
  }
  
  public String argumentTransclusion(final String name, final String id) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("dime");
    {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(id);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        _builder.append("-");
      }
    }
    _builder.append(id);
    _builder.append("-placeholder=\"");
    _builder.append(name);
    _builder.append("\"");
    return _builder.toString();
  }
  
  /**
   * Returns the static URL of the given button if one is present
   */
  public String getStaticURL(final Button button) {
    ButtonOptions _options = null;
    if (button!=null) {
      _options=button.getOptions();
    }
    return this.toStaticURL(_options);
  }
  
  public String toStaticURL(final ButtonOptions options) {
    String _xblockexpression = null;
    {
      if ((this.operator_equals(options, null) || StringExtensions.isNullOrEmpty(options.getStaticURL()))) {
        return null;
      }
      if ((options.getStaticURL().equals("/logout") || options.getStaticURL().equals("logout"))) {
        return "{{getLogoutURL}}";
      }
      String _staticURL = null;
      if (options!=null) {
        _staticURL=options.getStaticURL();
      }
      _xblockexpression = _staticURL;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the static URL of the given button if one is present
   */
  public String getStaticURL(final LinkSIB button) {
    ButtonOptions _options = null;
    if (button!=null) {
      _options=button.getOptions();
    }
    return this.toStaticURL(_options);
  }
  
  protected void _getIgnorableBranches(final SIB sib, final Map<String, String> ignorableBranches) {
  }
  
  protected void _getIgnorableBranches(final info.scce.dime.process.process.GUISIB sib, final Map<String, String> ignorableBranches) {
    final Function1<GUIBranch, String> _function = new Function1<GUIBranch, String>() {
      @Override
      public String apply(final GUIBranch it) {
        return it.getName();
      }
    };
    final Consumer<String> _function_1 = new Consumer<String>() {
      @Override
      public void accept(final String it) {
        ignorableBranches.put(it, it);
      }
    };
    IterableExtensions.<GUIBranch, String>map(this.getGUIBranchesMerged(sib.getGui()), _function).forEach(_function_1);
  }
  
  protected void _getIgnorableBranches(final info.scce.dime.process.process.ProcessSIB sib, final Map<String, String> ignorableBranches) {
    final Function1<EndSIB, String> _function = new Function1<EndSIB, String>() {
      @Override
      public String apply(final EndSIB it) {
        return it.getBranchName();
      }
    };
    final Consumer<String> _function_1 = new Consumer<String>() {
      @Override
      public void accept(final String it) {
        ignorableBranches.put(it, it);
      }
    };
    ListExtensions.<EndSIB, String>map(sib.getProMod().getEndSIBs(), _function).forEach(_function_1);
  }
  
  protected void _getIgnorableBranches(final AtomicSIB sib, final Map<String, String> ignorableBranches) {
    EObject _sib = sib.getSib();
    final Function1<info.scce.dime.siblibrary.Branch, String> _function = new Function1<info.scce.dime.siblibrary.Branch, String>() {
      @Override
      public String apply(final info.scce.dime.siblibrary.Branch it) {
        return it.getName();
      }
    };
    final Consumer<String> _function_1 = new Consumer<String>() {
      @Override
      public void accept(final String it) {
        ignorableBranches.put(it, it);
      }
    };
    ListExtensions.<info.scce.dime.siblibrary.Branch, String>map(((info.scce.dime.siblibrary.SIB) _sib).getBranches(), _function).forEach(_function_1);
  }
  
  protected void _getIgnorableBranches(final RetrieveCurrentUserSIB sib, final Map<String, String> ignorableBranches) {
    ignorableBranches.put("not authenticated", "not authenticated");
  }
  
  protected void _getIgnorableBranches(final RemoveFromListSIB sib, final Map<String, String> ignorableBranches) {
    ignorableBranches.put("not found", "not found");
  }
  
  protected void _getIgnorableBranches(final RetrieveOfTypeSIB sib, final Map<String, String> ignorableBranches) {
    ignorableBranches.put("none found", "none found");
  }
  
  public String defaultValue(final Object pv) {
    if (pv instanceof ComplexInputPort) {
      return _defaultValue((ComplexInputPort)pv);
    } else if (pv instanceof PrimitiveInputPort) {
      return _defaultValue((PrimitiveInputPort)pv);
    } else if (pv instanceof info.scce.dime.process.process.InputStatic) {
      return _defaultValue((info.scce.dime.process.process.InputStatic)pv);
    } else if (pv instanceof ComplexVariable) {
      return _defaultValue((ComplexVariable)pv);
    } else if (pv instanceof PrimitiveVariable) {
      return _defaultValue((PrimitiveVariable)pv);
    } else if (pv instanceof InputStatic) {
      return _defaultValue((InputStatic)pv);
    } else if (pv instanceof IO) {
      return _defaultValue((IO)pv);
    } else if (pv instanceof PrimitiveType) {
      return _defaultValue((PrimitiveType)pv);
    } else if (pv instanceof info.scce.dime.gui.gui.PrimitiveType) {
      return _defaultValue((info.scce.dime.gui.gui.PrimitiveType)pv);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(pv).toString());
    }
  }
  
  public void getIgnorableBranches(final SIB sib, final Map<String, String> ignorableBranches) {
    if (sib instanceof AtomicSIB) {
      _getIgnorableBranches((AtomicSIB)sib, ignorableBranches);
      return;
    } else if (sib instanceof info.scce.dime.process.process.GUISIB) {
      _getIgnorableBranches((info.scce.dime.process.process.GUISIB)sib, ignorableBranches);
      return;
    } else if (sib instanceof info.scce.dime.process.process.ProcessSIB) {
      _getIgnorableBranches((info.scce.dime.process.process.ProcessSIB)sib, ignorableBranches);
      return;
    } else if (sib instanceof RemoveFromListSIB) {
      _getIgnorableBranches((RemoveFromListSIB)sib, ignorableBranches);
      return;
    } else if (sib instanceof RetrieveCurrentUserSIB) {
      _getIgnorableBranches((RetrieveCurrentUserSIB)sib, ignorableBranches);
      return;
    } else if (sib instanceof RetrieveOfTypeSIB) {
      _getIgnorableBranches((RetrieveOfTypeSIB)sib, ignorableBranches);
      return;
    } else if (sib != null) {
      _getIgnorableBranches(sib, ignorableBranches);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(sib, ignorableBranches).toString());
    }
  }
}
