/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.gui.checks;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import de.jabc.cinco.meta.runtime.xapi.CodingExtension;
import graphmodel.Container;
import graphmodel.Edge;
import graphmodel.IdentifiableElement;
import graphmodel.ModelElementContainer;
import graphmodel.Node;
import info.scce.dime.checks.GUICheck;
import info.scce.dime.data.data.Type;
import info.scce.dime.data.helper.DataExtension;
import info.scce.dime.gui.gui.ComplexVariable;
import info.scce.dime.gui.gui.ExtensionContext;
import info.scce.dime.gui.gui.GUI;
import info.scce.dime.gui.gui.GUISIB;
import info.scce.dime.gui.gui.PrimitiveType;
import info.scce.dime.gui.gui.PrimitiveVariable;
import info.scce.dime.gui.gui.Variable;
import java.util.Arrays;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.jooq.lambda.Seq;
import org.jooq.lambda.tuple.Tuple2;

/**
 * The extension check is used to validate the signature
 * of the current GUI model, if it extends another GUI model.
 * 
 * @author zweihoff
 * @author neubauer
 */
@SuppressWarnings("all")
public class ExtensionCheck extends GUICheck {
  @Extension
  private DataExtension _dataExtension = DataExtension.getInstance();
  
  @Extension
  private CodingExtension _codingExtension = new CodingExtension();
  
  @Override
  public void check(final GUI model) {
    EList<ExtensionContext> _extensionContexts = null;
    if (model!=null) {
      _extensionContexts=model.getExtensionContexts();
    }
    ExtensionContext _head = null;
    if (_extensionContexts!=null) {
      _head=IterableExtensions.<ExtensionContext>head(_extensionContexts);
    }
    final ExtensionContext extensionContext = _head;
    EList<GUISIB> _gUISIBs = null;
    if (extensionContext!=null) {
      _gUISIBs=extensionContext.getGUISIBs();
    }
    final EList<GUISIB> guiSIBs = _gUISIBs;
    int _size = 0;
    if (guiSIBs!=null) {
      _size=guiSIBs.size();
    }
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      ModelElementContainer _elvis = null;
      if (extensionContext != null) {
        _elvis = extensionContext;
      } else {
        _elvis = model;
      }
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Only one GUI extension is allowed");
      this.addError(_elvis, _builder.toString());
      return;
    }
    GUISIB _head_1 = null;
    if (guiSIBs!=null) {
      _head_1=IterableExtensions.<GUISIB>head(guiSIBs);
    }
    if (_head_1!=null) {
      final Procedure1<GUISIB> _function = new Procedure1<GUISIB>() {
        @Override
        public void apply(final GUISIB it) {
          ExtensionCheck.this.checkSignature(it);
          ExtensionCheck.this.checkNoConnections(it);
          ExtensionCheck.this.checkNoSelfExtension(it);
        }
      };
      this._codingExtension.<GUISIB>with(_head_1, _function);
    }
  }
  
  private void checkNoConnections(final Node it) {
    boolean _noEdges = this.getNoEdges(it);
    boolean _not = (!_noEdges);
    if (_not) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("No edges allowed on extension");
      this.addError(it, _builder.toString());
    }
    if ((it instanceof Container)) {
      final Consumer<Node> _function = new Consumer<Node>() {
        @Override
        public void accept(final Node it) {
          ExtensionCheck.this.checkNoConnections(it);
        }
      };
      Iterables.<Node>filter(((Container)it).getModelElements(), Node.class).forEach(_function);
    }
  }
  
  private void checkNoSelfExtension(final GUISIB it) {
    GUI _gui = it.getGui();
    GUI _rootElement = it.getRootElement();
    boolean _equals = this.guiex.operator_equals(_gui, _rootElement);
    if (_equals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("No self extension allowed");
      this.addError(it, _builder.toString());
    }
  }
  
  private boolean getNoEdges(final Node it) {
    return (it.<Edge>getIncoming().isEmpty() && it.<Edge>getOutgoing().isEmpty());
  }
  
  private void checkSignature(final GUISIB parentGUISIB) {
    final GUI guiModel = parentGUISIB.getRootElement();
    final GUI parentGUI = parentGUISIB.getGui();
    final Function1<Variable, String> _function = new Function1<Variable, String>() {
      @Override
      public String apply(final Variable it) {
        return it.getName();
      }
    };
    final Map<String, Variable> requiredInputVariables = Seq.<Object, String, Variable>toMap(this._collectionExtension.<String, Variable>associateWithKey(this.guiex.inputVariables(parentGUI), _function));
    final Iterable<Variable> presentInputVariables = this.guiex.inputVariables(guiModel);
    final Function1<Variable, Variable> _function_1 = new Function1<Variable, Variable>() {
      @Override
      public Variable apply(final Variable it) {
        return requiredInputVariables.get(it.getName());
      }
    };
    final Predicate<Tuple2<Variable, Variable>> _function_2 = new Predicate<Tuple2<Variable, Variable>>() {
      @Override
      public boolean test(final Tuple2<Variable, Variable> it) {
        return ExtensionCheck.this.guiex.operator_notEquals(
          it.v1, null);
      }
    };
    final Predicate<Tuple2<Variable, Variable>> _function_3 = new Predicate<Tuple2<Variable, Variable>>() {
      @Override
      public boolean test(final Tuple2<Variable, Variable> it) {
        return ExtensionCheck.this.checkListStateEq(parentGUISIB, it.v1, it.v2);
      }
    };
    final Predicate<Tuple2<Variable, Variable>> _function_4 = new Predicate<Tuple2<Variable, Variable>>() {
      @Override
      public boolean test(final Tuple2<Variable, Variable> it) {
        return ExtensionCheck.this.checkVariable(parentGUISIB, it.v1, it.v2);
      }
    };
    final boolean noMatchingInput = this._collectionExtension.<Variable, Variable>associateWithKey(presentInputVariables, _function_1).filter(_function_2).filter(_function_3).filter(_function_4).isEmpty();
    if (noMatchingInput) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("No (valid) overriding input found for sub GUI \"");
      String _title = guiModel.getTitle();
      _builder.append(_title);
      _builder.append("\".");
      this.addError(parentGUISIB, _builder.toString());
    }
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final Variable v1, final Variable v2) {
    return false;
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final ComplexVariable v1, final ComplexVariable v2) {
    boolean _xifexpression = false;
    Type _dataType = v2.getDataType();
    Type _dataType_1 = v1.getDataType();
    boolean _equals = this.guiex.operator_equals(_dataType, _dataType_1);
    if (_equals) {
      boolean _xblockexpression = false;
      {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Overridden input \"");
        String _name = v1.getName();
        _builder.append(_name);
        _builder.append("\" has same type as overriding input. So it does not override for real.");
        this.addInfo(it, _builder.toString());
        _xblockexpression = false;
      }
      _xifexpression = _xblockexpression;
    } else {
      boolean _xifexpression_1 = false;
      boolean _isTypeOf = this._dataExtension.isTypeOf(v2.getDataType(), v1.getDataType());
      if (_isTypeOf) {
        _xifexpression_1 = true;
      } else {
        boolean _xblockexpression_1 = false;
        {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("The type of overriding input \"");
          String _name = v1.getName();
          _builder.append(_name);
          _builder.append(" is not a sub type of the overriden input\'s type.\"");
          this.addError(it, _builder.toString());
          _xblockexpression_1 = false;
        }
        _xifexpression_1 = _xblockexpression_1;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final ComplexVariable v1, final Variable v2) {
    boolean _xblockexpression = false;
    {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Overridden input \"");
      String _name = v1.getName();
      _builder.append(_name);
      _builder.append("\" represents a complex type, but overriding input not.");
      this.addError(it, _builder.toString());
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final Variable v1, final ComplexVariable v2) {
    boolean _xblockexpression = false;
    {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Overriding input \"");
      String _name = v1.getName();
      _builder.append(_name);
      _builder.append("\" represents a complex type, but overridden input not.");
      this.addError(it, _builder.toString());
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final PrimitiveVariable v1, final PrimitiveVariable v2) {
    boolean _xblockexpression = false;
    {
      PrimitiveType _dataType = v1.getDataType();
      PrimitiveType _dataType_1 = v2.getDataType();
      boolean _notEquals = (!Objects.equal(_dataType, _dataType_1));
      if (_notEquals) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Overridden input \"");
        String _name = v1.getName();
        _builder.append(_name);
        _builder.append("\" has not same primitive type as overriding input.");
        this.addError(it, _builder.toString());
      }
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("Overridden input \"");
      String _name_1 = v1.getName();
      _builder_1.append(_name_1);
      _builder_1.append("\" has same type as overriding input. So it does not override for real.");
      this.addInfo(it, _builder_1.toString());
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final PrimitiveVariable v1, final Variable v2) {
    boolean _xblockexpression = false;
    {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Overridden input \"");
      String _name = v1.getName();
      _builder.append(_name);
      _builder.append("\" represents a primitive type, but overriding input not.");
      this.addError(it, _builder.toString());
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  private boolean _checkVariable(final IdentifiableElement it, final Variable v1, final PrimitiveVariable v2) {
    boolean _xblockexpression = false;
    {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Overriding input \"");
      String _name = v2.getName();
      _builder.append(_name);
      _builder.append("\" represents a primitive type, but overridden input not.");
      this.addError(it, _builder.toString());
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
  
  private boolean checkListStateEq(final IdentifiableElement it, final Variable v1, final Variable v2) {
    boolean _isIsList = v1.isIsList();
    boolean _isIsList_1 = v2.isIsList();
    boolean _notEquals = (_isIsList != _isIsList_1);
    if (_notEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("List type status for input \"");
      String _name = v1.getName();
      _builder.append(_name);
      _builder.append("\" differ.");
      this.addError(it, _builder.toString());
      return false;
    }
    return true;
  }
  
  private boolean checkVariable(final IdentifiableElement it, final Variable v1, final Variable v2) {
    if (v1 instanceof ComplexVariable
         && v2 instanceof ComplexVariable) {
      return _checkVariable(it, (ComplexVariable)v1, (ComplexVariable)v2);
    } else if (v1 instanceof PrimitiveVariable
         && v2 instanceof PrimitiveVariable) {
      return _checkVariable(it, (PrimitiveVariable)v1, (PrimitiveVariable)v2);
    } else if (v1 instanceof ComplexVariable
         && v2 != null) {
      return _checkVariable(it, (ComplexVariable)v1, v2);
    } else if (v1 instanceof PrimitiveVariable
         && v2 != null) {
      return _checkVariable(it, (PrimitiveVariable)v1, v2);
    } else if (v1 != null
         && v2 instanceof ComplexVariable) {
      return _checkVariable(it, v1, (ComplexVariable)v2);
    } else if (v1 != null
         && v2 instanceof PrimitiveVariable) {
      return _checkVariable(it, v1, (PrimitiveVariable)v2);
    } else if (v1 != null
         && v2 != null) {
      return _checkVariable(it, v1, v2);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it, v1, v2).toString());
    }
  }
}
