/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.gui.checks;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import graphmodel.Node;
import info.scce.dime.checks.AbstractCheck;
import info.scce.dime.data.data.Type;
import info.scce.dime.data.helper.DataExtension;
import info.scce.dime.gui.gui.ComplexAttribute;
import info.scce.dime.gui.gui.ComplexInputPort;
import info.scce.dime.gui.gui.ComplexVariable;
import info.scce.dime.gui.gui.ControlSIB;
import info.scce.dime.gui.gui.FORSIB;
import info.scce.dime.gui.gui.IFSIB;
import info.scce.dime.gui.gui.ISSIB;
import info.scce.dime.gui.gui.Read;
import info.scce.dime.gui.gui.SecuritySIB;
import info.scce.dime.gui.mcam.adapter.GUIAdapter;
import info.scce.dime.gui.mcam.adapter.GUIId;
import info.scce.dime.process.process.EndSIB;
import info.scce.dime.process.process.ProcessType;
import java.util.List;
import java.util.function.Function;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.jooq.lambda.Seq;

/**
 * The event check is used to validate the event listeners
 * of all listener contexts of a GUI model
 * @author zweihoff
 */
@SuppressWarnings("all")
public class ControlSIBCheck extends AbstractCheck<GUIId, GUIAdapter> {
  private GUIAdapter adapter;
  
  @Extension
  protected DataExtension _dataExtension = DataExtension.getInstance();
  
  /**
   * Checks, if the events in given GUI model
   * are valid.
   */
  @Override
  public void doExecute(final GUIAdapter arg0) {
    this.adapter = arg0;
    List<GUIId> _entityIds = this.adapter.getEntityIds();
    for (final GUIId id : _entityIds) {
      {
        Object obj = id.getElement();
        if ((obj instanceof ISSIB)) {
          this.checkControlSIB(id, ((ControlSIB)obj));
          this.checkISSIB(id, ((ISSIB)obj));
        }
        if ((obj instanceof FORSIB)) {
          this.checkControlSIB(id, ((ControlSIB)obj));
          this.checkForSIB(id, ((FORSIB)obj));
        }
        if ((obj instanceof IFSIB)) {
          this.checkControlSIB(id, ((ControlSIB)obj));
        }
        if ((obj instanceof SecuritySIB)) {
          this.checkSecuritySIB(id, ((SecuritySIB)obj));
        }
      }
    }
  }
  
  public void checkISSIB(final GUIId id, final ISSIB issib) {
    boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<ComplexInputPort>filter(issib.getIOs(), ComplexInputPort.class));
    boolean _not = (!_isEmpty);
    if (_not) {
      final ComplexInputPort inputPort = ((ComplexInputPort[])Conversions.unwrapArray((Iterables.<ComplexInputPort>filter(issib.getIOs(), ComplexInputPort.class)), ComplexInputPort.class))[0];
      boolean _isIsList = inputPort.isIsList();
      if (_isIsList) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("complex input port has to be not list");
        this.addError(id, _builder.toString());
      }
      boolean _isEmpty_1 = inputPort.<Read>getIncoming(Read.class).isEmpty();
      boolean _not_1 = (!_isEmpty_1);
      if (_not_1) {
        final Node dataSource = inputPort.<Read>getIncoming(Read.class).get(0).getSourceElement();
        final Function<Type, String> _function = new Function<Type, String>() {
          @Override
          public String apply(final Type it) {
            return it.getName();
          }
        };
        Seq<String> _map = this._dataExtension.getSuperTypes(inputPort.getDataType()).<String>map(_function);
        String _name = inputPort.getDataType().getName();
        final String superTypeNames = (_map + _name);
        if ((dataSource instanceof ComplexVariable)) {
          final String typeName = ((ComplexVariable)dataSource).getDataType().getName();
          boolean _contains = superTypeNames.contains(typeName);
          boolean _not_2 = (!_contains);
          if (_not_2) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("connected type is not in same hierarchy");
            this.addError(id, _builder_1.toString());
          }
        } else {
          if ((dataSource instanceof ComplexAttribute)) {
            final String typeName_1 = ((ComplexAttribute)dataSource).getAttribute().getDataType().getName();
            boolean _contains_1 = superTypeNames.contains(typeName_1);
            boolean _not_3 = (!_contains_1);
            if (_not_3) {
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append("connected type is not in same hierarchy");
              this.addError(id, _builder_2.toString());
            }
          } else {
            StringConcatenation _builder_3 = new StringConcatenation();
            _builder_3.append("only complex objects can be validated");
            this.addError(id, _builder_3.toString());
          }
        }
      }
    }
  }
  
  public void checkForSIB(final GUIId id, final FORSIB forsib) {
    int _from = forsib.getFrom();
    boolean _lessThan = (_from < 0);
    if (_lessThan) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("start index has to be greater than 0");
      this.addError(id, _builder.toString());
    }
    int _to = forsib.getTo();
    boolean _lessThan_1 = (_to < (-1));
    if (_lessThan_1) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("end index has to be greater than -1");
      this.addError(id, _builder_1.toString());
    }
    if (((forsib.getFrom() > forsib.getTo()) && (forsib.getTo() > (-1)))) {
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("start index has to less than end index");
      this.addError(id, _builder_2.toString());
    }
    boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<ComplexInputPort>filter(forsib.getIOs(), ComplexInputPort.class));
    boolean _not = (!_isEmpty);
    if (_not) {
      boolean _isIsList = (((ComplexInputPort[])Conversions.unwrapArray((Iterables.<ComplexInputPort>filter(forsib.getIOs(), ComplexInputPort.class)), ComplexInputPort.class))[0]).isIsList();
      boolean _not_1 = (!_isIsList);
      if (_not_1) {
        StringConcatenation _builder_3 = new StringConcatenation();
        _builder_3.append("complex input port has to be list");
        this.addError(id, _builder_3.toString());
      }
      boolean _isEmpty_1 = (((ComplexInputPort[])Conversions.unwrapArray((Iterables.<ComplexInputPort>filter(forsib.getIOs(), ComplexInputPort.class)), ComplexInputPort.class))[0]).<Read>getIncoming(Read.class).isEmpty();
      boolean _not_2 = (!_isEmpty_1);
      if (_not_2) {
        final Node v = (((ComplexInputPort[])Conversions.unwrapArray((Iterables.<ComplexInputPort>filter(forsib.getIOs(), ComplexInputPort.class)), ComplexInputPort.class))[0]).<Read>getIncoming(Read.class).get(0).getSourceElement();
        if ((v instanceof ComplexVariable)) {
          boolean _isIsList_1 = ((ComplexVariable)v).isIsList();
          boolean _not_3 = (!_isIsList_1);
          if (_not_3) {
            StringConcatenation _builder_4 = new StringConcatenation();
            _builder_4.append("iteration source has to be list");
            this.addError(id, _builder_4.toString());
          }
        }
        if ((v instanceof ComplexAttribute)) {
          boolean _isIsList_2 = ((ComplexAttribute)v).getAttribute().isIsList();
          boolean _not_4 = (!_isIsList_2);
          if (_not_4) {
            StringConcatenation _builder_5 = new StringConcatenation();
            _builder_5.append("iteration source has to be list");
            this.addError(id, _builder_5.toString());
          }
        }
      }
    }
  }
  
  public void checkControlSIB(final GUIId id, final ControlSIB issib) {
    int _size = issib.getIOs().size();
    boolean _notEquals = (_size != 1);
    if (_notEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("one complex input port required");
      this.addError(id, _builder.toString());
    } else {
      boolean _isEmpty = issib.getIOs().get(0).<Read>getIncoming(Read.class).isEmpty();
      if (_isEmpty) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("read edge required");
        this.addError(id, _builder_1.toString());
      }
    }
  }
  
  public void checkSecuritySIB(final GUIId id, final SecuritySIB secSib) {
    info.scce.dime.process.process.Process _proMod = secSib.getProMod();
    ProcessType _processType = ((info.scce.dime.process.process.Process) _proMod).getProcessType();
    boolean _notEquals = (!Objects.equal(_processType, ProcessType.BASIC));
    if (_notEquals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("only interactbale process supported");
      this.addError(id, _builder.toString());
    }
    info.scce.dime.process.process.Process _proMod_1 = secSib.getProMod();
    final Function1<EndSIB, Boolean> _function = new Function1<EndSIB, Boolean>() {
      @Override
      public Boolean apply(final EndSIB n) {
        return Boolean.valueOf(n.getBranchName().equals("granted"));
      }
    };
    boolean _exists = IterableExtensions.<EndSIB>exists(((info.scce.dime.process.process.Process) _proMod_1).getEndSIBs(), _function);
    boolean _not = (!_exists);
    if (_not) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("granted end SIB required");
      this.addError(id, _builder_1.toString());
    }
    info.scce.dime.process.process.Process _proMod_2 = secSib.getProMod();
    final Function1<EndSIB, Boolean> _function_1 = new Function1<EndSIB, Boolean>() {
      @Override
      public Boolean apply(final EndSIB n) {
        return Boolean.valueOf(n.getBranchName().equals("denied"));
      }
    };
    boolean _exists_1 = IterableExtensions.<EndSIB>exists(((info.scce.dime.process.process.Process) _proMod_2).getEndSIBs(), _function_1);
    boolean _not_1 = (!_exists_1);
    if (_not_1) {
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("denied end SIB required");
      this.addError(id, _builder_2.toString());
    }
    info.scce.dime.process.process.Process _proMod_3 = secSib.getProMod();
    int _size = ((info.scce.dime.process.process.Process) _proMod_3).getEndSIBs().size();
    boolean _notEquals_1 = (_size != 2);
    if (_notEquals_1) {
      StringConcatenation _builder_3 = new StringConcatenation();
      _builder_3.append("only denied and granted end SIB allowed");
      this.addError(id, _builder_3.toString());
    }
  }
  
  @Override
  public void init() {
  }
}
