/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.gui.actions;

import graphmodel.ModelElementContainer;
import info.scce.dime.api.DIMECustomAction;
import info.scce.dime.gui.gui.BooleanInputStatic;
import info.scce.dime.gui.gui.ControlSIB;
import info.scce.dime.gui.gui.EventListener;
import info.scce.dime.gui.gui.InputStatic;
import info.scce.dime.gui.gui.IntegerInputStatic;
import info.scce.dime.gui.gui.PrimitiveInputPort;
import info.scce.dime.gui.gui.PrimitiveType;
import info.scce.dime.gui.gui.RealInputStatic;
import info.scce.dime.gui.gui.SIB;
import info.scce.dime.gui.gui.TextInputStatic;
import info.scce.dime.gui.gui.TimestampInputStatic;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Context menu action to convert a static primitive input port of a GUI SIB or GUI plug in SIB
 * to a none static input port of the same data type.
 * The context menu entry is available for static primitive input ports.
 * @author zweihoff
 */
@SuppressWarnings("all")
public class StaticToPrimitivePort<T extends InputStatic> extends DIMECustomAction<InputStatic> {
  @Override
  public String getName() {
    return "Convert Static Value to Input";
  }
  
  /**
   * Creates a new primitive input port with the corresponding type
   * of the static primitive port data type and deleted the static primitive port afterwards.
   */
  @Override
  public void execute(final InputStatic inputStatic) {
    try {
      int x = inputStatic.getX();
      int y = inputStatic.getY();
      String name = inputStatic.getName();
      ModelElementContainer container = inputStatic.getContainer();
      inputStatic.delete();
      PrimitiveType pType = null;
      if ((inputStatic instanceof IntegerInputStatic)) {
        pType = PrimitiveType.INTEGER;
      } else {
        if ((inputStatic instanceof RealInputStatic)) {
          pType = PrimitiveType.REAL;
        } else {
          if ((inputStatic instanceof TextInputStatic)) {
            pType = PrimitiveType.TEXT;
          } else {
            if ((inputStatic instanceof TimestampInputStatic)) {
              pType = PrimitiveType.TIMESTAMP;
            } else {
              if ((inputStatic instanceof BooleanInputStatic)) {
                pType = PrimitiveType.BOOLEAN;
              } else {
                throw new IllegalStateException("missing implementation for static port type");
              }
            }
          }
        }
      }
      PrimitiveInputPort cInputPort = null;
      if ((container instanceof SIB)) {
        cInputPort = ((SIB) container).newPrimitiveInputPort(x, y);
      }
      if ((container instanceof EventListener)) {
        cInputPort = ((EventListener)container).newPrimitiveInputPort(x, y);
      }
      if ((container instanceof ControlSIB)) {
        cInputPort = ((ControlSIB)container).newPrimitiveInputPort(x, y);
      }
      cInputPort.setName(name);
      cInputPort.setDataType(pType);
      cInputPort.highlight();
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        e.printStackTrace();
        return;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
}
