/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.data.helper;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import de.jabc.cinco.meta.runtime.xapi.CollectionExtension;
import de.jabc.cinco.meta.runtime.xapi.GraphModelExtension;
import info.scce.dime.data.data.AbstractType;
import info.scce.dime.data.data.Attribute;
import info.scce.dime.data.data.ComplexAttribute;
import info.scce.dime.data.data.ExtensionAttribute;
import info.scce.dime.data.data.Inheritance;
import info.scce.dime.data.data.PrimitiveAttribute;
import info.scce.dime.data.data.PrimitiveType;
import info.scce.dime.data.data.ReferencedBidirectionalAttribute;
import info.scce.dime.data.data.ReferencedComplexAttribute;
import info.scce.dime.data.data.ReferencedEnumType;
import info.scce.dime.data.data.ReferencedExtensionAttribute;
import info.scce.dime.data.data.ReferencedPrimitiveAttribute;
import info.scce.dime.data.data.ReferencedType;
import info.scce.dime.data.data.ReferencedUserAttribute;
import info.scce.dime.data.data.ReferencedUserType;
import info.scce.dime.data.data.Type;
import info.scce.dime.data.data.UserAttribute;
import info.scce.dime.data.data.UserType;
import info.scce.dime.process.process.BooleanInputStatic;
import info.scce.dime.process.process.ComplexInputPort;
import info.scce.dime.process.process.Input;
import info.scce.dime.process.process.InputStatic;
import info.scce.dime.process.process.IntegerInputStatic;
import info.scce.dime.process.process.PrimitiveInputPort;
import info.scce.dime.process.process.RealInputStatic;
import info.scce.dime.process.process.TextInputStatic;
import info.scce.dime.process.process.TimestampInputStatic;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.jooq.lambda.Seq;
import org.jooq.lambda.tuple.Tuple2;

@SuppressWarnings("all")
public class DataExtension {
  @Extension
  protected GraphModelExtension _graphModelExtension = new GraphModelExtension();
  
  @Extension
  protected CollectionExtension _collectionExtension = new CollectionExtension();
  
  /**
   * Singleton pattern (keep this class stateless!)
   */
  private static DataExtension _instance;
  
  public static DataExtension getInstance() {
    DataExtension _elvis = null;
    if (DataExtension._instance != null) {
      _elvis = DataExtension._instance;
    } else {
      DataExtension _dataExtension = new DataExtension();
      DataExtension __instance = (DataExtension._instance = _dataExtension);
      _elvis = __instance;
    }
    return _elvis;
  }
  
  private DataExtension() {
  }
  
  /**
   * Sorts the given sequence of types topologically by the inheritance
   * relation ascending. This means, the leaves in the inheritance DAG (we support
   * multiple inheritance) come before their parent types and so on.
   * 
   * If you iterate through such a list from left to right, more specific types are
   * evaluated before less specific types, so that no "shadowing" occurs.
   * 
   * @param it the sequence of types.
   * @return a sorted sequence containing all sub types of the given type in the same data model.
   */
  public List<Type> sortTopologically(final Seq<Type> it) {
    final Comparator<Type> _function = new Comparator<Type>() {
      @Override
      public int compare(final Type a, final Type b) {
        int _xifexpression = (int) 0;
        boolean _equals = DataExtension.this._graphModelExtension.operator_equals(a, b);
        if (_equals) {
          _xifexpression = 0;
        } else {
          int _xifexpression_1 = (int) 0;
          boolean _contains = DataExtension.this.getSuperTypes(a).contains(b);
          if (_contains) {
            _xifexpression_1 = (-1);
          } else {
            int _xifexpression_2 = (int) 0;
            boolean _contains_1 = DataExtension.this.getSuperTypes(b).contains(a);
            if (_contains_1) {
              _xifexpression_2 = 1;
            } else {
              _xifexpression_2 = 0;
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        return _xifexpression;
      }
    };
    return it.sorted(_function).toList();
  }
  
  /**
   * Collects all sub types of the given type in the data model of the type.
   * 
   * @param it the (super) type. All sub types in the same data model are returned.
   * @return a sequence containing all sub types of the given type in the same data model.
   */
  public Seq<Type> getKnownSubTypes(final Type it) {
    return Seq.<Type>seq(this.getKnownSubTypesRecursive(it)).distinct();
  }
  
  private List<Type> getKnownSubTypesRecursive(final Type it) {
    final Function1<Inheritance, List<Type>> _function = new Function1<Inheritance, List<Type>>() {
      @Override
      public List<Type> apply(final Inheritance it) {
        return DataExtension.this.getKnownSubTypesRecursive(it.getSourceElement());
      }
    };
    final Function<Type, Type> _function_1 = new Function<Type, Type>() {
      @Override
      public Type apply(final Type it) {
        return DataExtension.this.getOriginalType(it);
      }
    };
    return Seq.<Type>of(it).concat(
      IterableExtensions.<Inheritance, Type>flatMap(Seq.<Inheritance>seq(it.<Inheritance>getIncoming(Inheritance.class)), _function)).<Type>map(_function_1).toList();
  }
  
  /**
   * Collects all attributes of type, including the ones that are inherited from parent types
   * and returns them.
   * 
   * Does not add those inherited attributes that are specialized/overriden, i.e., it only adds the
   * most specific one.
   * 
   * @param type the type to search for attributes.
   * @return the sequence of leaf attributes.
   */
  public Seq<Attribute> getInheritedAttributeSeq(final Type type) {
    Seq<Attribute> _xblockexpression = null;
    {
      final Type originalType = this.getOriginalType(type);
      final Function<Type, Type> _function = new Function<Type, Type>() {
        @Override
        public Type apply(final Type it) {
          return DataExtension.this.getOriginalType(it);
        }
      };
      final Seq<Type> types = Seq.<Type>of(originalType).concat(this.getSuperTypes(originalType).<Type>map(_function));
      final Function1<Type, EList<Attribute>> _function_1 = new Function1<Type, EList<Attribute>>() {
        @Override
        public EList<Attribute> apply(final Type it) {
          return it.getAttributes();
        }
      };
      final Function1<Attribute, String> _function_2 = new Function1<Attribute, String>() {
        @Override
        public String apply(final Attribute it) {
          return it.getName();
        }
      };
      final Collection<Attribute> attrs = IterableExtensions.<String, Attribute>toMap(IterableExtensions.<Type, Attribute>flatMap(types, _function_1), _function_2).values();
      final Tuple2<Seq<Attribute>, Seq<Attribute>> attrDupes = this._collectionExtension.<Attribute>duplicate(attrs);
      final Tuple2<Seq<Attribute>, Seq<Attribute>> attrDupes2 = attrDupes.v1.duplicate();
      final Seq<Attribute> attrs1 = attrDupes.v2;
      final Seq<Attribute> attrs2 = attrDupes2.v1;
      final Seq<Attribute> attrs3 = attrDupes2.v2;
      final Function1<ComplexAttribute, ComplexAttribute> _function_3 = new Function1<ComplexAttribute, ComplexAttribute>() {
        @Override
        public ComplexAttribute apply(final ComplexAttribute it) {
          return it.getSuperAttr();
        }
      };
      final Function1<PrimitiveAttribute, PrimitiveAttribute> _function_4 = new Function1<PrimitiveAttribute, PrimitiveAttribute>() {
        @Override
        public PrimitiveAttribute apply(final PrimitiveAttribute it) {
          return it.getSuperAttr();
        }
      };
      final Predicate<Attribute> _function_5 = new Predicate<Attribute>() {
        @Override
        public boolean test(final Attribute it) {
          return (!(it instanceof ComplexAttribute));
        }
      };
      final Predicate<Attribute> _function_6 = new Predicate<Attribute>() {
        @Override
        public boolean test(final Attribute it) {
          return (!(it instanceof PrimitiveAttribute));
        }
      };
      _xblockexpression = Seq.<Attribute>empty().concat(
        this._collectionExtension.<ComplexAttribute>removeByAssociation(attrs1.<ComplexAttribute>ofType(ComplexAttribute.class), _function_3)).concat(
        this._collectionExtension.<PrimitiveAttribute>removeByAssociation(attrs2.<PrimitiveAttribute>ofType(PrimitiveAttribute.class), _function_4)).concat(
        attrs3.filter(_function_5).filter(_function_6));
    }
    return _xblockexpression;
  }
  
  /**
   * Collects all attributes of type, including the ones that are inherited from parent types
   * and adds them to collectedAttributes.
   * 
   * Does not add those inherited attributes that are specialized, i.e., it only adds the
   * most specific one.
   * 
   * @param type the type to search attributes for.
   * @return the list of attributes (as described above).
   */
  public List<Attribute> getInheritedAttributes(final Type type) {
    return this.getInheritedAttributeSeq(type).toList();
  }
  
  /**
   * Retrieves all super types of the given type, recursively.
   * 
   * @param the type to search super types for.
   * @return the sequence of super types recursively until the root type.
   */
  public Seq<Type> getSuperTypes(final Type type) {
    return Seq.<Type>seq(Iterables.<Type>filter(this._graphModelExtension.findSuccessorsVia(type, Inheritance.class), Type.class));
  }
  
  /**
   * Retrieves root super types of the given type, recursively.
   * If the type has no super type, the type itself is returned.
   * 
   * @param the type to search root types for.
   * @return the sequence of root super types.
   */
  public Seq<Type> getRootTypes(final Type type) {
    final Predicate<Type> _function = new Predicate<Type>() {
      @Override
      public boolean test(final Type it) {
        return DataExtension.this.getDirectSuperTypes(it).isEmpty();
      }
    };
    return Seq.<Type>of(type).concat(this.getSuperTypes(type)).filter(_function);
  }
  
  /**
   * Retrieves all direct super types (<strong>not</strong> super types of the super types).
   * 
   * @param teh type to search direct super types for.
   * @return the sequence of direct super types.
   */
  public Seq<Type> getDirectSuperTypes(final Type type) {
    final Function<Inheritance, Type> _function = new Function<Inheritance, Type>() {
      @Override
      public Type apply(final Inheritance it) {
        return DataExtension.this.getOriginalType(it.getTargetElement());
      }
    };
    return Seq.<Inheritance>seq(this.getOriginalType(type).<Inheritance>getOutgoing(Inheritance.class)).<Type>map(_function);
  }
  
  /**
   * Checks whether the given type is an abstract type.
   * 
   * @param type the type to check.
   * @return whether the given type is an abstract type.
   */
  public boolean isAbstract(final Type type) {
    Type _originalType = this.getOriginalType(type);
    return (_originalType instanceof AbstractType);
  }
  
  /**
   * Checks whether the given <code>type</code> is a sub type of the given <code>superType</code>.
   * 
   * @param type the type to check whether it is a sub type.
   * @param superType the potential super type.
   * @return whether the given <code>type</code> is a sub type of the given <code>superType</code>.
   */
  public boolean isTypeOf(final Type type, final Type superType) {
    boolean _xblockexpression = false;
    {
      final Type originalType = this.getOriginalType(type);
      final Type originalSuperType = this.getOriginalType(superType);
      boolean _xifexpression = false;
      boolean _equals = this._graphModelExtension.operator_equals(originalType, originalSuperType);
      if (_equals) {
        return true;
      } else {
        _xifexpression = this.getSuperTypes(originalType).contains(originalSuperType);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public boolean isBooleanAttribute(final Attribute attr) {
    return this.hasPrimitiveType(attr, PrimitiveType.BOOLEAN);
  }
  
  public boolean isFileAttribute(final Attribute attr) {
    return this.hasPrimitiveType(attr, PrimitiveType.FILE);
  }
  
  public boolean isIntegerAttribute(final Attribute attr) {
    return this.hasPrimitiveType(attr, PrimitiveType.INTEGER);
  }
  
  public boolean isRealAttribute(final Attribute attr) {
    return this.hasPrimitiveType(attr, PrimitiveType.REAL);
  }
  
  public boolean isTextAttribute(final Attribute attr) {
    return this.hasPrimitiveType(attr, PrimitiveType.TEXT);
  }
  
  public boolean isTimestampAttribute(final Attribute attr) {
    return this.hasPrimitiveType(attr, PrimitiveType.TIMESTAMP);
  }
  
  public boolean hasPrimitiveType(final Attribute attr, final PrimitiveType type) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (attr instanceof PrimitiveAttribute) {
      _matched=true;
      PrimitiveType _dataType = ((PrimitiveAttribute)attr).getDataType();
      _switchResult = Objects.equal(_dataType, type);
    }
    return _switchResult;
  }
  
  /**
   * Returns types of extension attribute
   */
  public Input getExtensionAttributePort(final ExtensionAttribute ea) {
    Input _xblockexpression = null;
    {
      info.scce.dime.process.process.Process _process = ea.getProcess();
      final info.scce.dime.process.process.Process p = ((info.scce.dime.process.process.Process) _process);
      if ((p.getEndSIBs().isEmpty() || p.getEndSIBs().get(0).getInputs().isEmpty())) {
        return null;
      }
      _xblockexpression = p.getEndSIBs().get(0).getInputs().get(0);
    }
    return _xblockexpression;
  }
  
  public Input getExtensionAttributePort(final ReferencedExtensionAttribute ea) {
    return this.getExtensionAttributePort(ea.getReferencedAttribute());
  }
  
  public Type getComplexExtensionAttributeType(final ExtensionAttribute ea) {
    Object _xblockexpression = null;
    {
      final Input input = this.getExtensionAttributePort(ea);
      if (((input != null) && (input instanceof ComplexInputPort))) {
        return this.getOriginalType(((ComplexInputPort) input).getDataType());
      }
      _xblockexpression = null;
    }
    return ((Type)_xblockexpression);
  }
  
  public Type getComplexExtensionAttributeType(final ReferencedExtensionAttribute ea) {
    return this.getComplexExtensionAttributeType(ea.getReferencedAttribute());
  }
  
  public PrimitiveType getPrimitiveExtensionAttributeType(final ExtensionAttribute ea) {
    Object _xblockexpression = null;
    {
      final Input port = this.getExtensionAttributePort(ea);
      if ((port == null)) {
        return null;
      }
      if ((port instanceof PrimitiveInputPort)) {
        return this.toData(((PrimitiveInputPort)port).getDataType());
      }
      if ((port instanceof InputStatic)) {
        return this.toData(((InputStatic)port));
      }
      _xblockexpression = null;
    }
    return ((PrimitiveType)_xblockexpression);
  }
  
  public PrimitiveType getPrimitiveExtensionAttributeType(final ReferencedExtensionAttribute ea) {
    return this.getPrimitiveExtensionAttributeType(ea.getReferencedAttribute());
  }
  
  public Type getComplexDataType(final Attribute attribute) {
    Object _xblockexpression = null;
    {
      if ((attribute instanceof ExtensionAttribute)) {
        return this.getComplexExtensionAttributeType(((ExtensionAttribute)attribute));
      }
      if ((attribute instanceof ComplexAttribute)) {
        return ((ComplexAttribute)attribute).getDataType();
      }
      _xblockexpression = null;
    }
    return ((Type)_xblockexpression);
  }
  
  public PrimitiveType getPrimitiveDataType(final Attribute attribute) {
    Object _xblockexpression = null;
    {
      if ((attribute instanceof ExtensionAttribute)) {
        return this.getPrimitiveExtensionAttributeType(((ExtensionAttribute)attribute));
      }
      if ((attribute instanceof PrimitiveAttribute)) {
        return ((PrimitiveAttribute)attribute).getDataType();
      }
      _xblockexpression = null;
    }
    return ((PrimitiveType)_xblockexpression);
  }
  
  public boolean isPrimitive(final Attribute attribute) {
    if ((attribute instanceof PrimitiveAttribute)) {
      return true;
    }
    if ((attribute instanceof ExtensionAttribute)) {
      PrimitiveType _primitiveExtensionAttributeType = this.getPrimitiveExtensionAttributeType(((ExtensionAttribute)attribute));
      return (_primitiveExtensionAttributeType != null);
    }
    return false;
  }
  
  public boolean isComplex(final Attribute attribute) {
    if ((attribute instanceof ComplexAttribute)) {
      return true;
    }
    if ((attribute instanceof ExtensionAttribute)) {
      Type _complexExtensionAttributeType = this.getComplexExtensionAttributeType(((ExtensionAttribute)attribute));
      return (_complexExtensionAttributeType != null);
    }
    return false;
  }
  
  /**
   * Get the original type of this (potentially) referenced type.
   * 
   * @param type the (potentially) referenced type.
   * @return the original type if the type is a referenced type or itself.
   */
  public Type getOriginalType(final Type type) {
    Type _xblockexpression = null;
    {
      if ((type == null)) {
        throw new IllegalStateException("Type null");
      }
      Type _switchResult = null;
      final Type it = type;
      boolean _matched = false;
      if (it instanceof ReferencedType) {
        _matched=true;
        _switchResult = this.getOriginalType(((ReferencedType)it).getReferencedType());
      }
      if (!_matched) {
        if (it instanceof ReferencedUserType) {
          _matched=true;
          _switchResult = this.getOriginalType(((ReferencedUserType)it).getReferencedType());
        }
      }
      if (!_matched) {
        if (it instanceof ReferencedEnumType) {
          _matched=true;
          _switchResult = this.getOriginalType(((ReferencedEnumType)it).getReferencedType());
        }
      }
      if (!_matched) {
        _switchResult = it;
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  /**
   * Maps a primitive type (i.e. the enum value) of gui models to
   * primitive types of data models.
   * 
   * @param it the primitive type enum value of gui.
   * @return the corresponding primitive type enum value of data.
   */
  public PrimitiveType toData(final info.scce.dime.gui.gui.PrimitiveType it) {
    PrimitiveType _switchResult = null;
    if (it != null) {
      switch (it) {
        case BOOLEAN:
          _switchResult = PrimitiveType.BOOLEAN;
          break;
        case FILE:
          _switchResult = PrimitiveType.FILE;
          break;
        case INTEGER:
          _switchResult = PrimitiveType.INTEGER;
          break;
        case REAL:
          _switchResult = PrimitiveType.REAL;
          break;
        case TEXT:
          _switchResult = PrimitiveType.TEXT;
          break;
        case TIMESTAMP:
          _switchResult = PrimitiveType.TIMESTAMP;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  /**
   * Maps a primitive type (i.e. the enum value) of process models to
   * primitive types of data models.
   * 
   * @param it the primitive type enum value of process.
   * @return the corresponding primitive type enum value of data.
   */
  public PrimitiveType toData(final info.scce.dime.process.process.PrimitiveType it) {
    PrimitiveType _switchResult = null;
    if (it != null) {
      switch (it) {
        case BOOLEAN:
          _switchResult = PrimitiveType.BOOLEAN;
          break;
        case FILE:
          _switchResult = PrimitiveType.FILE;
          break;
        case INTEGER:
          _switchResult = PrimitiveType.INTEGER;
          break;
        case REAL:
          _switchResult = PrimitiveType.REAL;
          break;
        case TEXT:
          _switchResult = PrimitiveType.TEXT;
          break;
        case TIMESTAMP:
          _switchResult = PrimitiveType.TIMESTAMP;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  /**
   * Maps a primitive type (i.e. the enum value) of gui models to
   * primitive types of data models.
   * 
   * @param it the primitive type enum value of gui.
   * @return the corresponding primitive type enum value of data.
   */
  public PrimitiveType toData(final InputStatic it) {
    PrimitiveType _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(it, BooleanInputStatic.class)) {
      _matched=true;
      _switchResult = PrimitiveType.BOOLEAN;
    }
    if (!_matched) {
      if (Objects.equal(it, IntegerInputStatic.class)) {
        _matched=true;
        _switchResult = PrimitiveType.INTEGER;
      }
    }
    if (!_matched) {
      if (Objects.equal(it, RealInputStatic.class)) {
        _matched=true;
        _switchResult = PrimitiveType.REAL;
      }
    }
    if (!_matched) {
      if (Objects.equal(it, TextInputStatic.class)) {
        _matched=true;
        _switchResult = PrimitiveType.TEXT;
      }
    }
    if (!_matched) {
      if (Objects.equal(it, TimestampInputStatic.class)) {
        _matched=true;
        _switchResult = PrimitiveType.TIMESTAMP;
      }
    }
    return _switchResult;
  }
  
  /**
   * Get the original type of this (potentially) referenced attribute.
   * 
   * @param attr the (potentially) referenced attribute.
   * @return the original type if the attribute is a referenced attribute or itself.
   */
  public Type getOriginalType(final ComplexAttribute attribute) {
    return this.getOriginalType(attribute.getDataType());
  }
  
  /**
   * Get the original attribute of this (potentially) referenced attribute.
   * 
   * @param attr the (potentially) referenced attribute.
   * @return the original attribute if the given attribute is a referenced attribute or itself.
   */
  public Attribute getOriginalAttribute(final Attribute attr) {
    Attribute _switchResult = null;
    final Attribute it = attr;
    boolean _matched = false;
    if (it instanceof ReferencedComplexAttribute) {
      _matched=true;
      _switchResult = this.getOriginalAttribute(((ReferencedComplexAttribute)it).getReferencedAttribute());
    }
    if (!_matched) {
      if (it instanceof ReferencedBidirectionalAttribute) {
        _matched=true;
        _switchResult = this.getOriginalAttribute(((ReferencedBidirectionalAttribute)it).getReferencedAttribute());
      }
    }
    if (!_matched) {
      if (it instanceof ReferencedUserAttribute) {
        _matched=true;
        _switchResult = this.getOriginalAttribute(((ReferencedUserAttribute)it).getReferencedAttribute());
      }
    }
    if (!_matched) {
      if (it instanceof ReferencedPrimitiveAttribute) {
        _matched=true;
        _switchResult = this.getOriginalAttribute(((ReferencedPrimitiveAttribute)it).getReferencedAttribute());
      }
    }
    if (!_matched) {
      if (it instanceof ReferencedExtensionAttribute) {
        _matched=true;
        _switchResult = this.getOriginalAttribute(((ReferencedExtensionAttribute)it).getReferencedAttribute());
      }
    }
    if (!_matched) {
      _switchResult = it;
    }
    return _switchResult;
  }
  
  /**
   * Returns the type of the given attribute.
   * 
   * @param attr the attribute.
   * @return the attribute's type.
   */
  public Type getType(final Attribute attr) {
    final Type mec = attr.getContainer();
    if ((mec instanceof Type)) {
      return this.getOriginalType(mec);
    }
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Attribute ");
    String _name = null;
    if (attr!=null) {
      _name=attr.getName();
    }
    _builder.append(_name);
    _builder.append(" without type container found... should never happen.");
    throw new IllegalStateException(_builder.toString());
  }
  
  public Type getConcreteUserType(final UserType it) {
    return IterableExtensions.<UserAttribute>head(it.getUserAttributes()).getDataType();
  }
}
