/**
 * -
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.api.modelgen;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import de.jabc.cinco.meta.core.ge.style.generator.runtime.layout.EdgeLayout;
import de.jabc.cinco.meta.core.ge.style.generator.runtime.layout.EdgeLayoutUtils;
import de.jabc.cinco.meta.core.referenceregistry.ReferenceRegistry;
import de.jabc.cinco.meta.runtime.CincoRuntimeBaseClass;
import de.jabc.cinco.meta.runtime.xapi.NIOExtension;
import graphmodel.Container;
import graphmodel.Edge;
import graphmodel.ModelElementContainer;
import graphmodel.Node;
import graphmodel.internal.InternalEdge;
import graphmodel.internal.InternalModelElement;
import graphmodel.internal._Point;
import info.scce.dime.data.data.Attribute;
import info.scce.dime.data.data.BidirectionalAttribute;
import info.scce.dime.data.data.ConcreteType;
import info.scce.dime.data.data.Type;
import info.scce.dime.data.helper.DataExtension;
import info.scce.dime.process.actions.PrimitivePortToStatic;
import info.scce.dime.process.factory.ProcessFactory;
import info.scce.dime.process.helper.PortUtils;
import info.scce.dime.process.helper.ProcessExtension;
import info.scce.dime.process.helper.SIBLayoutUtils;
import info.scce.dime.process.process.BooleanInputStatic;
import info.scce.dime.process.process.Branch;
import info.scce.dime.process.process.BranchConnector;
import info.scce.dime.process.process.ComplexAttribute;
import info.scce.dime.process.process.ComplexAttributeConnector;
import info.scce.dime.process.process.ComplexInputPort;
import info.scce.dime.process.process.ComplexOutputPort;
import info.scce.dime.process.process.ComplexVariable;
import info.scce.dime.process.process.DataContext;
import info.scce.dime.process.process.DataFlow;
import info.scce.dime.process.process.DataFlowSource;
import info.scce.dime.process.process.DataFlowTarget;
import info.scce.dime.process.process.IO;
import info.scce.dime.process.process.InputPort;
import info.scce.dime.process.process.InputStatic;
import info.scce.dime.process.process.IntegerInputStatic;
import info.scce.dime.process.process.OutputPort;
import info.scce.dime.process.process.PrimitiveAttribute;
import info.scce.dime.process.process.PrimitiveInputPort;
import info.scce.dime.process.process.PrimitiveOutputPort;
import info.scce.dime.process.process.PrimitiveType;
import info.scce.dime.process.process.PrimitiveVariable;
import info.scce.dime.process.process.ProcessType;
import info.scce.dime.process.process.RealInputStatic;
import info.scce.dime.process.process.SIB;
import info.scce.dime.process.process.TextInputStatic;
import info.scce.dime.process.process.TimestampInputStatic;
import info.scce.dime.process.process.Variable;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.ArrayExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Extension API for convenient process model generation.
 * 
 * @author Steve Bosselmann
 */
@SuppressWarnings("all")
public class ProcessModelGenerationLanguage extends CincoRuntimeBaseClass {
  @Extension
  private DataExtension _dataExtension = DataExtension.getInstance();
  
  @Extension
  private ProcessExtension _processExtension = new ProcessExtension();
  
  @Extension
  private EdgeLayoutUtils _edgeLayoutUtils = new EdgeLayoutUtils();
  
  @Extension
  private PortUtils _portUtils = new PortUtils();
  
  @Extension
  private NIOExtension _nIOExtension = new NIOExtension();
  
  public static final int VARIABLE_DEFAULT_HEIGHT = 25;
  
  public static final int DATACONTEXT_PADDING_TOP = 23;
  
  public static final int VARIABLE_MARGIN_X = 10;
  
  public static final int VARIABLE_MARGIN_Y = 20;
  
  public static final int VARIABLE_PADDING_TOP = 23;
  
  public static final int VARIABLE_UNFOLDED_ATTRIBUTE_MARGIN_TOP = 20;
  
  public static final int ATTRIBUTE_DEFAULT_HEIGHT = 25;
  
  public static final int ATTRIBUTE_MARGIN_X = 10;
  
  public static final int ATTRIBUTE_MARGIN_Y = 5;
  
  public static final int BRANCH_TO_SIB_OFFSET = 30;
  
  public static final int SIB_TO_BRANCH_OFFSET = 10;
  
  private ProcessFactory processFactory = ProcessFactory.eINSTANCE;
  
  public info.scce.dime.process.process.Process createBasicProcess(final Path outlet, final String name) {
    return this.createBasicProcess(outlet, name, true, true);
  }
  
  public info.scce.dime.process.process.Process createBasicProcess(final Path outlet, final String name, final boolean runPostCreate, final boolean save) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(name);
    _builder.append(".process");
    final Path path = outlet.resolve(_builder.toString());
    final URI uri = this._nIOExtension.toEURI(path);
    final Resource res = new ResourceSetImpl().createResource(uri);
    info.scce.dime.process.process.Process _createProcess = this.processFactory.createProcess();
    final Procedure1<info.scce.dime.process.process.Process> _function = new Procedure1<info.scce.dime.process.process.Process>() {
      @Override
      public void apply(final info.scce.dime.process.process.Process it) {
        it.setProcessType(ProcessType.BASIC);
        EcoreUtil.setID(it, EcoreUtil.generateUUID());
      }
    };
    final info.scce.dime.process.process.Process process = ObjectExtensions.<info.scce.dime.process.process.Process>operator_doubleArrow(_createProcess, _function);
    res.getContents().add(process.getInternalElement_());
    if (runPostCreate) {
      this.processFactory.postCreates(process);
    }
    if (save) {
      process.save();
    }
    return process;
  }
  
  public Attribute getAttribute(final Type propType, final String name) {
    final Function1<Attribute, Boolean> _function = new Function1<Attribute, Boolean>() {
      @Override
      public Boolean apply(final Attribute it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return IterableExtensions.<Attribute>head(IterableExtensions.<Attribute>filter(this._dataExtension.getInheritedAttributes(propType), _function));
  }
  
  public info.scce.dime.process.process.Attribute getAttribute(final ComplexVariable compVar, final String name) {
    final Function1<info.scce.dime.process.process.Attribute, Boolean> _function = new Function1<info.scce.dime.process.process.Attribute, Boolean>() {
      @Override
      public Boolean apply(final info.scce.dime.process.process.Attribute it) {
        boolean _switchResult = false;
        boolean _matched = false;
        if (it instanceof ComplexAttribute) {
          _matched=true;
          String _name = ((ComplexAttribute)it).getAttribute().getName();
          _switchResult = Objects.equal(_name, name);
        }
        if (!_matched) {
          if (it instanceof PrimitiveAttribute) {
            _matched=true;
            String _name = ((PrimitiveAttribute)it).getAttribute().getName();
            _switchResult = Objects.equal(_name, name);
          }
        }
        if (!_matched) {
          _switchResult = false;
        }
        return Boolean.valueOf(_switchResult);
      }
    };
    return this._graphModelExtension.<info.scce.dime.process.process.Attribute>findThe(compVar, info.scce.dime.process.process.Attribute.class, _function);
  }
  
  public info.scce.dime.data.data.ComplexAttribute getComplexAttribute(final Type propType, final String name) {
    final Function1<Attribute, Boolean> _function = new Function1<Attribute, Boolean>() {
      @Override
      public Boolean apply(final Attribute it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return IterableExtensions.<info.scce.dime.data.data.ComplexAttribute>head(Iterables.<info.scce.dime.data.data.ComplexAttribute>filter(IterableExtensions.<Attribute>filter(this._dataExtension.getInheritedAttributes(propType), _function), info.scce.dime.data.data.ComplexAttribute.class));
  }
  
  public info.scce.dime.data.data.PrimitiveAttribute getPrimitiveAttribute(final Type propType, final String name) {
    final Function1<Attribute, Boolean> _function = new Function1<Attribute, Boolean>() {
      @Override
      public Boolean apply(final Attribute it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return IterableExtensions.<info.scce.dime.data.data.PrimitiveAttribute>head(Iterables.<info.scce.dime.data.data.PrimitiveAttribute>filter(IterableExtensions.<Attribute>filter(this._dataExtension.getInheritedAttributes(propType), _function), info.scce.dime.data.data.PrimitiveAttribute.class));
  }
  
  public BidirectionalAttribute getBidirectionalAttribute(final Type propType, final String name) {
    final Function1<Attribute, Boolean> _function = new Function1<Attribute, Boolean>() {
      @Override
      public Boolean apply(final Attribute it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return IterableExtensions.<BidirectionalAttribute>head(Iterables.<BidirectionalAttribute>filter(IterableExtensions.<Attribute>filter(this._dataExtension.getInheritedAttributes(propType), _function), BidirectionalAttribute.class));
  }
  
  protected info.scce.dime.process.process.Attribute _addAttribute(final ComplexVariable compVar, final String attrName) {
    info.scce.dime.process.process.Attribute _xblockexpression = null;
    {
      final Type parentType = compVar.getDataType();
      final Attribute attr = this.getAttribute(parentType, attrName);
      _xblockexpression = this.addAttribute(compVar, attr);
    }
    return _xblockexpression;
  }
  
  protected info.scce.dime.process.process.Attribute _addAttribute(final ComplexVariable it, final info.scce.dime.data.data.PrimitiveAttribute attr) {
    final PrimitiveAttribute newAttr = it.newPrimitiveAttribute(attr, ProcessModelGenerationLanguage.ATTRIBUTE_MARGIN_X, this.beneathBottomAttribute(it), this.maxAttributeWidth(it), ProcessModelGenerationLanguage.ATTRIBUTE_DEFAULT_HEIGHT);
    this.increaseIfNecessary(it);
    return newAttr;
  }
  
  protected info.scce.dime.process.process.Attribute _addAttribute(final ComplexVariable it, final info.scce.dime.data.data.ComplexAttribute attr) {
    final ComplexAttribute newAttr = it.newComplexAttribute(attr, ProcessModelGenerationLanguage.ATTRIBUTE_MARGIN_X, this.beneathBottomAttribute(it), this.maxAttributeWidth(it), ProcessModelGenerationLanguage.ATTRIBUTE_DEFAULT_HEIGHT);
    this.increaseIfNecessary(it);
    return newAttr;
  }
  
  public ComplexVariable unfoldAttribute(final ComplexVariable compVar, final String attrName) {
    ComplexVariable _xblockexpression = null;
    {
      final Type parentType = compVar.getDataType();
      final info.scce.dime.data.data.ComplexAttribute attr = this.getComplexAttribute(parentType, attrName);
      final Type type = attr.getDataType();
      final DataContext dataContext = compVar.getContainer();
      int _x = compVar.getX();
      int _bottom = this.getBottom(compVar);
      int _plus = (_bottom + ProcessModelGenerationLanguage.VARIABLE_UNFOLDED_ATTRIBUTE_MARGIN_TOP);
      ComplexVariable _newComplexVariable = dataContext.newComplexVariable(type, _x, _plus, compVar.getWidth(), ProcessModelGenerationLanguage.VARIABLE_DEFAULT_HEIGHT);
      final Procedure1<ComplexVariable> _function = new Procedure1<ComplexVariable>() {
        @Override
        public void apply(final ComplexVariable it) {
          it.setName(attr.getName());
          ComplexAttributeConnector _newComplexAttributeConnector = compVar.newComplexAttributeConnector(it);
          final Procedure1<ComplexAttributeConnector> _function = new Procedure1<ComplexAttributeConnector>() {
            @Override
            public void apply(final ComplexAttributeConnector it) {
              it.setAttributeName(attrName);
            }
          };
          ObjectExtensions.<ComplexAttributeConnector>operator_doubleArrow(_newComplexAttributeConnector, _function);
        }
      };
      _xblockexpression = ObjectExtensions.<ComplexVariable>operator_doubleArrow(_newComplexVariable, _function);
    }
    return _xblockexpression;
  }
  
  public ComplexVariable addVariable(final DataContext ctx, final ConcreteType type) {
    return this.addVariable(ctx, type, StringExtensions.toFirstLower(type.getName()));
  }
  
  public ComplexVariable addVariable(final DataContext it, final ConcreteType type, final String varName) {
    ComplexVariable _newComplexVariable = it.newComplexVariable(type, 
      ProcessModelGenerationLanguage.VARIABLE_MARGIN_X, this.beneathBottomVariable(it), this.maxVariableWidth(it), ProcessModelGenerationLanguage.VARIABLE_DEFAULT_HEIGHT);
    final Procedure1<ComplexVariable> _function = new Procedure1<ComplexVariable>() {
      @Override
      public void apply(final ComplexVariable it) {
        it.setName(varName);
      }
    };
    return ObjectExtensions.<ComplexVariable>operator_doubleArrow(_newComplexVariable, _function);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final Type dataType) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, dataType);
    return ((ComplexInputPort) _addInput);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final info.scce.dime.data.data.PrimitiveAttribute dataAttr) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, dataAttr);
    return ((PrimitiveInputPort) _addInput);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final PrimitiveAttribute attr) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, attr);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(attr.getAttribute().getName());
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final info.scce.dime.data.data.ComplexAttribute attr) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, attr);
    return ((ComplexInputPort) _addInput);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final ComplexAttribute attr) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, this._processExtension.getComplexType(attr));
    final Procedure1<ComplexInputPort> _function = new Procedure1<ComplexInputPort>() {
      @Override
      public void apply(final ComplexInputPort it) {
        it.setName(attr.getAttribute().getName());
      }
    };
    return ObjectExtensions.<ComplexInputPort>operator_doubleArrow(
      ((ComplexInputPort) _addInput), _function);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final PrimitiveVariable primVar) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, this._processExtension.getPrimitiveType(primVar));
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(primVar.getName());
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  protected InputPort _addInputPort(final Container dataFlowTarget, final ComplexVariable compVar) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, this._processExtension.getComplexType(compVar));
    final Procedure1<ComplexInputPort> _function = new Procedure1<ComplexInputPort>() {
      @Override
      public void apply(final ComplexInputPort it) {
        it.setName(compVar.getName());
      }
    };
    return ObjectExtensions.<ComplexInputPort>operator_doubleArrow(
      ((ComplexInputPort) _addInput), _function);
  }
  
  public PrimitiveInputPort addBooleanInputPort(final Container dataFlowTarget, final String name) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, PrimitiveType.BOOLEAN);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(name);
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  public PrimitiveInputPort addIntegerInputPort(final Container dataFlowTarget, final String name) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, PrimitiveType.INTEGER);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(name);
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  public PrimitiveInputPort addRealInputPort(final Container dataFlowTarget, final String name) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, PrimitiveType.REAL);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(name);
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  public PrimitiveInputPort addTimestampInputPort(final Container dataFlowTarget, final String name) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, PrimitiveType.TIMESTAMP);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(name);
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  public PrimitiveInputPort addTextInputPort(final Container dataFlowTarget, final String name) {
    IO _addInput = PortUtils.addInput(dataFlowTarget, PrimitiveType.TEXT);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(name);
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addInput), _function);
  }
  
  public InputStatic addStaticInputPort(final SIB sib, final info.scce.dime.data.data.PrimitiveType type, final String name, final Object value) {
    InputStatic _switchResult = null;
    if (type != null) {
      switch (type) {
        case BOOLEAN:
          BooleanInputStatic _addStaticBooleanInputPort = this.addStaticBooleanInputPort(sib, name);
          final Procedure1<BooleanInputStatic> _function = new Procedure1<BooleanInputStatic>() {
            @Override
            public void apply(final BooleanInputStatic it) {
              it.setValue((((Boolean) value)).booleanValue());
            }
          };
          _switchResult = ObjectExtensions.<BooleanInputStatic>operator_doubleArrow(_addStaticBooleanInputPort, _function);
          break;
        case FILE:
          throw new UnsupportedOperationException("Static input ports of type FILE are not supported");
        case INTEGER:
          IntegerInputStatic _addStaticIntegerInputPort = this.addStaticIntegerInputPort(sib, name);
          final Procedure1<IntegerInputStatic> _function_1 = new Procedure1<IntegerInputStatic>() {
            @Override
            public void apply(final IntegerInputStatic it) {
              it.setValue((((Long) value)).longValue());
            }
          };
          _switchResult = ObjectExtensions.<IntegerInputStatic>operator_doubleArrow(_addStaticIntegerInputPort, _function_1);
          break;
        case REAL:
          RealInputStatic _addStaticRealInputPort = this.addStaticRealInputPort(sib, name);
          final Procedure1<RealInputStatic> _function_2 = new Procedure1<RealInputStatic>() {
            @Override
            public void apply(final RealInputStatic it) {
              it.setValue((((Double) value)).doubleValue());
            }
          };
          _switchResult = ObjectExtensions.<RealInputStatic>operator_doubleArrow(_addStaticRealInputPort, _function_2);
          break;
        case TEXT:
          TextInputStatic _addStaticTextInputPort = this.addStaticTextInputPort(sib, name);
          final Procedure1<TextInputStatic> _function_3 = new Procedure1<TextInputStatic>() {
            @Override
            public void apply(final TextInputStatic it) {
              it.setValue(((String) value));
            }
          };
          _switchResult = ObjectExtensions.<TextInputStatic>operator_doubleArrow(_addStaticTextInputPort, _function_3);
          break;
        case TIMESTAMP:
          TimestampInputStatic _addStaticTimestampInputPort = this.addStaticTimestampInputPort(sib, name);
          final Procedure1<TimestampInputStatic> _function_4 = new Procedure1<TimestampInputStatic>() {
            @Override
            public void apply(final TimestampInputStatic it) {
              it.setValue((((Long) value)).longValue());
            }
          };
          _switchResult = ObjectExtensions.<TimestampInputStatic>operator_doubleArrow(_addStaticTimestampInputPort, _function_4);
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  public BooleanInputStatic addStaticBooleanInputPort(final DataFlowTarget dataFlowTarget, final String name) {
    BooleanInputStatic _newBooleanInputStatic = dataFlowTarget.newBooleanInputStatic(0, 0);
    final Procedure1<BooleanInputStatic> _function = new Procedure1<BooleanInputStatic>() {
      @Override
      public void apply(final BooleanInputStatic it) {
        it.setName(name);
        SIBLayoutUtils.resizeAndLayout(dataFlowTarget);
      }
    };
    return ObjectExtensions.<BooleanInputStatic>operator_doubleArrow(_newBooleanInputStatic, _function);
  }
  
  public IntegerInputStatic addStaticIntegerInputPort(final DataFlowTarget dataFlowTarget, final String name) {
    IntegerInputStatic _newIntegerInputStatic = dataFlowTarget.newIntegerInputStatic(0, 0);
    final Procedure1<IntegerInputStatic> _function = new Procedure1<IntegerInputStatic>() {
      @Override
      public void apply(final IntegerInputStatic it) {
        it.setName(name);
        SIBLayoutUtils.resizeAndLayout(dataFlowTarget);
      }
    };
    return ObjectExtensions.<IntegerInputStatic>operator_doubleArrow(_newIntegerInputStatic, _function);
  }
  
  public RealInputStatic addStaticRealInputPort(final DataFlowTarget dataFlowTarget, final String name) {
    RealInputStatic _newRealInputStatic = dataFlowTarget.newRealInputStatic(0, 0);
    final Procedure1<RealInputStatic> _function = new Procedure1<RealInputStatic>() {
      @Override
      public void apply(final RealInputStatic it) {
        it.setName(name);
        SIBLayoutUtils.resizeAndLayout(dataFlowTarget);
      }
    };
    return ObjectExtensions.<RealInputStatic>operator_doubleArrow(_newRealInputStatic, _function);
  }
  
  public TextInputStatic addStaticTextInputPort(final DataFlowTarget dataFlowTarget, final String name) {
    TextInputStatic _newTextInputStatic = dataFlowTarget.newTextInputStatic(0, 0);
    final Procedure1<TextInputStatic> _function = new Procedure1<TextInputStatic>() {
      @Override
      public void apply(final TextInputStatic it) {
        it.setName(name);
        SIBLayoutUtils.resizeAndLayout(dataFlowTarget);
      }
    };
    return ObjectExtensions.<TextInputStatic>operator_doubleArrow(_newTextInputStatic, _function);
  }
  
  public TimestampInputStatic addStaticTimestampInputPort(final DataFlowTarget dataFlowTarget, final String name) {
    TimestampInputStatic _newTimestampInputStatic = dataFlowTarget.newTimestampInputStatic(0, 0);
    final Procedure1<TimestampInputStatic> _function = new Procedure1<TimestampInputStatic>() {
      @Override
      public void apply(final TimestampInputStatic it) {
        it.setName(name);
        SIBLayoutUtils.resizeAndLayout(dataFlowTarget);
      }
    };
    return ObjectExtensions.<TimestampInputStatic>operator_doubleArrow(_newTimestampInputStatic, _function);
  }
  
  public PrimitiveInputPort addTextOutputPort(final DataFlowSource dataFlowSource, final String name) {
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, PrimitiveType.TEXT);
    final Procedure1<PrimitiveInputPort> _function = new Procedure1<PrimitiveInputPort>() {
      @Override
      public void apply(final PrimitiveInputPort it) {
        it.setName(name);
      }
    };
    return ObjectExtensions.<PrimitiveInputPort>operator_doubleArrow(
      ((PrimitiveInputPort) _addOutput), _function);
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final Type dataType) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, dataType);
    final ComplexOutputPort port = ((ComplexOutputPort) _addOutput);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final info.scce.dime.data.data.PrimitiveAttribute dataAttr) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, dataAttr);
    final PrimitiveOutputPort port = ((PrimitiveOutputPort) _addOutput);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final info.scce.dime.data.data.ComplexAttribute attr) {
    final Function1<DataFlowSource, OutputPort> _function = new Function1<DataFlowSource, OutputPort>() {
      @Override
      public OutputPort apply(final DataFlowSource it) {
        OutputPort _addOutput = ProcessModelGenerationLanguage.this._portUtils.addOutput(dataFlowSource, attr.getDataType());
        return ((ComplexOutputPort) _addOutput);
      }
    };
    return this.addOuputPort(dataFlowSource, _function);
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final PrimitiveAttribute attr) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, this._processExtension.getPrimitiveType(attr));
    final Procedure1<PrimitiveOutputPort> _function = new Procedure1<PrimitiveOutputPort>() {
      @Override
      public void apply(final PrimitiveOutputPort it) {
        it.setName(attr.getAttribute().getName());
      }
    };
    final PrimitiveOutputPort port = ObjectExtensions.<PrimitiveOutputPort>operator_doubleArrow(((PrimitiveOutputPort) _addOutput), _function);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final ComplexAttribute attr) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, this._processExtension.getComplexType(attr));
    final Procedure1<ComplexOutputPort> _function = new Procedure1<ComplexOutputPort>() {
      @Override
      public void apply(final ComplexOutputPort it) {
        it.setName(attr.getAttribute().getName());
      }
    };
    final ComplexOutputPort port = ObjectExtensions.<ComplexOutputPort>operator_doubleArrow(((ComplexOutputPort) _addOutput), _function);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final PrimitiveVariable primVar) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, this._processExtension.getPrimitiveType(primVar));
    final Procedure1<PrimitiveOutputPort> _function = new Procedure1<PrimitiveOutputPort>() {
      @Override
      public void apply(final PrimitiveOutputPort it) {
        it.setName(primVar.getName());
      }
    };
    final PrimitiveOutputPort port = ObjectExtensions.<PrimitiveOutputPort>operator_doubleArrow(((PrimitiveOutputPort) _addOutput), _function);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final ComplexVariable compVar) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    OutputPort _addOutput = this._portUtils.addOutput(dataFlowSource, this._processExtension.getComplexType(compVar));
    final Procedure1<ComplexOutputPort> _function = new Procedure1<ComplexOutputPort>() {
      @Override
      public void apply(final ComplexOutputPort it) {
        it.setName(compVar.getName());
      }
    };
    final ComplexOutputPort port = ObjectExtensions.<ComplexOutputPort>operator_doubleArrow(((ComplexOutputPort) _addOutput), _function);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final PrimitiveInputPort input) {
    final Function1<DataFlowSource, OutputPort> _function = new Function1<DataFlowSource, OutputPort>() {
      @Override
      public OutputPort apply(final DataFlowSource it) {
        OutputPort _addOutput = ProcessModelGenerationLanguage.this._portUtils.addOutput(dataFlowSource, ProcessModelGenerationLanguage.this._processExtension.getPrimitiveType(input));
        final Procedure1<OutputPort> _function = new Procedure1<OutputPort>() {
          @Override
          public void apply(final OutputPort it) {
            it.setName(input.getName());
          }
        };
        return ObjectExtensions.<OutputPort>operator_doubleArrow(_addOutput, _function);
      }
    };
    return this.addOuputPort(dataFlowSource, _function);
  }
  
  protected OutputPort _addOutputPort(final DataFlowSource dataFlowSource, final ComplexInputPort input) {
    final Function1<DataFlowSource, OutputPort> _function = new Function1<DataFlowSource, OutputPort>() {
      @Override
      public OutputPort apply(final DataFlowSource it) {
        OutputPort _addOutput = ProcessModelGenerationLanguage.this._portUtils.addOutput(dataFlowSource, ProcessModelGenerationLanguage.this._processExtension.getComplexType(input));
        final Procedure1<ComplexOutputPort> _function = new Procedure1<ComplexOutputPort>() {
          @Override
          public void apply(final ComplexOutputPort it) {
            it.setName(input.getName());
          }
        };
        return ObjectExtensions.<ComplexOutputPort>operator_doubleArrow(
          ((ComplexOutputPort) _addOutput), _function);
      }
    };
    return this.addOuputPort(dataFlowSource, _function);
  }
  
  public OutputPort addOuputPort(final DataFlowSource dataFlowSource, final Function1<? super DataFlowSource, ? extends OutputPort> portProducer) {
    final int oldW = dataFlowSource.getWidth();
    final int oldH = dataFlowSource.getHeight();
    final OutputPort port = portProducer.apply(dataFlowSource);
    int _width = dataFlowSource.getWidth();
    final int deltaW = (_width - oldW);
    int _height = dataFlowSource.getHeight();
    final int deltaH = (_height - oldH);
    if (((deltaW != 0) || (deltaH != 0))) {
      this.moveSuccessors(dataFlowSource, deltaW, deltaH);
    }
    return port;
  }
  
  public OutputPort addOutputPort(final DataFlowSource branch, final info.scce.dime.data.data.PrimitiveType type, final String name) {
    final Function1<DataFlowSource, OutputPort> _function = new Function1<DataFlowSource, OutputPort>() {
      @Override
      public OutputPort apply(final DataFlowSource it) {
        PrimitiveOutputPort _newPrimitiveOutputPort = branch.newPrimitiveOutputPort(0, 0);
        final Procedure1<PrimitiveOutputPort> _function = new Procedure1<PrimitiveOutputPort>() {
          @Override
          public void apply(final PrimitiveOutputPort it) {
            it.setName(name);
            it.setIsList(false);
            it.setDataType(PortUtils.toPrimitiveType(type));
          }
        };
        return ObjectExtensions.<PrimitiveOutputPort>operator_doubleArrow(_newPrimitiveOutputPort, _function);
      }
    };
    return this.addOuputPort(branch, _function);
  }
  
  protected DataFlow _dataFlowTo(final ComplexAttribute attr, final ComplexInputPort port) {
    return attr.newComplexRead(port);
  }
  
  protected DataFlow _dataFlowTo(final PrimitiveAttribute attr, final PrimitiveInputPort port) {
    return attr.newPrimitiveRead(port);
  }
  
  protected DataFlow _dataFlowTo(final ComplexVariable compVar, final ComplexInputPort port) {
    return compVar.newComplexRead(port);
  }
  
  protected DataFlow _dataFlowTo(final PrimitiveVariable primVar, final PrimitiveInputPort port) {
    return primVar.newPrimitiveRead(port);
  }
  
  protected DataFlow _dataFlowTo(final PrimitiveOutputPort outPort, final PrimitiveVariable primVar) {
    return outPort.newPrimitiveUpdate(primVar);
  }
  
  protected DataFlow _dataFlowTo(final ComplexOutputPort outPort, final ComplexVariable compVar) {
    return outPort.newComplexUpdate(compVar);
  }
  
  protected DataFlow _dataFlowTo(final PrimitiveOutputPort outPort, final PrimitiveAttribute primAttr) {
    return outPort.newPrimitiveUpdate(primAttr);
  }
  
  protected DataFlow _dataFlowTo(final PrimitiveOutputPort outPort, final PrimitiveInputPort inPort) {
    return outPort.newPrimitiveDirectDataFlow(inPort);
  }
  
  protected DataFlow _dataFlowTo(final ComplexOutputPort outPort, final ComplexAttribute compAttr) {
    return outPort.newComplexUpdate(compAttr);
  }
  
  protected DataFlow _dataFlowTo(final ComplexOutputPort outPort, final ComplexInputPort inPort) {
    return outPort.newComplexDirectDataFlow(inPort);
  }
  
  protected DataFlow _dataFlowFrom(final ComplexAttribute attr, final ComplexOutputPort port) {
    return port.newComplexUpdate(attr);
  }
  
  protected DataFlow _dataFlowFrom(final PrimitiveAttribute attr, final PrimitiveOutputPort port) {
    return port.newPrimitiveUpdate(attr);
  }
  
  protected DataFlow _dataFlowFrom(final ComplexVariable compVar, final ComplexOutputPort port) {
    return port.newComplexUpdate(compVar);
  }
  
  protected DataFlow _dataFlowFrom(final PrimitiveVariable primVar, final PrimitiveOutputPort port) {
    return port.newPrimitiveUpdate(primVar);
  }
  
  protected DataFlow _dataFlowFrom(final PrimitiveInputPort inPort, final PrimitiveVariable primVar) {
    return primVar.newPrimitiveRead(inPort);
  }
  
  protected DataFlow _dataFlowFrom(final ComplexInputPort inPort, final ComplexVariable compVar) {
    return compVar.newComplexRead(inPort);
  }
  
  protected DataFlow _dataFlowFrom(final PrimitiveInputPort inPort, final PrimitiveAttribute primAttr) {
    return primAttr.newPrimitiveRead(inPort);
  }
  
  protected DataFlow _dataFlowFrom(final ComplexInputPort inPort, final ComplexAttribute compAttr) {
    return compAttr.newComplexRead(inPort);
  }
  
  protected DataFlow _dataFlowFrom(final ComplexInputPort inPort, final ComplexOutputPort outPort) {
    return outPort.newComplexDirectDataFlow(inPort);
  }
  
  public void setSuccessor(final DataFlowSource branch, final DataFlowTarget sib) {
    branch.newControlFlow(sib);
    this.alignBeneathOf(sib, branch);
  }
  
  public void setPredecessor(final DataFlowTarget sib, final DataFlowSource branch) {
    branch.newControlFlow(sib);
    this.alignBeneathOf(sib, branch);
  }
  
  public InputPort getInputPort(final DataFlowTarget sib, final String name) {
    final Function1<InputPort, Boolean> _function = new Function1<InputPort, Boolean>() {
      @Override
      public Boolean apply(final InputPort it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return this._graphModelExtension.<InputPort>findThe(sib, InputPort.class, _function);
  }
  
  public OutputPort getOutputPort(final DataFlowSource branch, final String name) {
    final Function1<OutputPort, Boolean> _function = new Function1<OutputPort, Boolean>() {
      @Override
      public Boolean apply(final OutputPort it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return this._graphModelExtension.<OutputPort>findThe(branch, OutputPort.class, _function);
  }
  
  public Branch getBranch(final DataFlowTarget sib, final String name) {
    final Function1<Branch, Boolean> _function = new Function1<Branch, Boolean>() {
      @Override
      public Boolean apply(final Branch it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    return IterableExtensions.<Branch>head(IterableExtensions.<Branch>filter(Iterables.<Branch>filter(this._graphModelExtension.<BranchConnector>findTargetsOf(sib, BranchConnector.class), Branch.class), _function));
  }
  
  public void keepBranches(final DataFlowTarget sib, final String... names) {
    final Function1<Branch, Boolean> _function = new Function1<Branch, Boolean>() {
      @Override
      public Boolean apply(final Branch it) {
        boolean _contains = ArrayExtensions.contains(names, it.getName());
        return Boolean.valueOf((!_contains));
      }
    };
    final Consumer<Branch> _function_1 = new Consumer<Branch>() {
      @Override
      public void accept(final Branch it) {
        it.delete();
      }
    };
    IterableExtensions.<Branch>filter(Iterables.<Branch>filter(this._graphModelExtension.<BranchConnector>findTargetsOf(sib, BranchConnector.class), Branch.class), _function).forEach(_function_1);
  }
  
  public void layoutMiddleToCenter(final Edge it) {
    final Runnable _function = new Runnable() {
      @Override
      public void run() {
        it.addBendpoint(
          ProcessModelGenerationLanguage.this._edgeLayoutUtils.getTopCenter(it.getTargetElement()).x, 
          ProcessModelGenerationLanguage.this._edgeLayoutUtils.getMiddleLeft(it.getSourceElement()).y);
      }
    };
    this._workbenchExtension.transact(it, _function);
  }
  
  public void increaseIfNecessary(final Container it) {
    this.increaseIfNecessaryRight(it);
    this.increaseIfNecessaryBottom(it);
  }
  
  protected void _increaseIfNecessaryBottom(final DataContext container) {
    this.increaseIfNecessaryBottom(container, ProcessModelGenerationLanguage.VARIABLE_MARGIN_Y);
  }
  
  protected void _increaseIfNecessaryBottom(final Variable container) {
    this.increaseIfNecessaryBottom(container, ProcessModelGenerationLanguage.ATTRIBUTE_MARGIN_Y);
  }
  
  public void increaseIfNecessaryBottom(final Container container, final int padding) {
    final Node botNode = this.getBottomChild(container);
    boolean _notEquals = this._graphModelExtension.operator_notEquals(botNode, null);
    if (_notEquals) {
      int _width = container.getWidth();
      int _bottom = this.getBottom(botNode);
      int _plus = (_bottom + padding);
      container.resize(_width, _plus);
      ModelElementContainer _container = container.getContainer();
      final ModelElementContainer it = _container;
      boolean _matched = false;
      if (it instanceof Container) {
        _matched=true;
        this.increaseIfNecessaryBottom(((Container)it));
      }
    }
  }
  
  protected void _increaseIfNecessaryRight(final DataContext container) {
    this.increaseIfNecessaryRight(container, ProcessModelGenerationLanguage.VARIABLE_MARGIN_X);
  }
  
  protected void _increaseIfNecessaryRight(final Variable container) {
    this.increaseIfNecessaryRight(container, ProcessModelGenerationLanguage.ATTRIBUTE_MARGIN_X);
  }
  
  public void increaseIfNecessaryRight(final Container container, final int padding) {
    final Node rightNode = this.getRightChild(container);
    boolean _notEquals = this._graphModelExtension.operator_notEquals(rightNode, null);
    if (_notEquals) {
      int _width = container.getWidth();
      int _right = this.getRight(rightNode);
      int _plus = (_right + padding);
      container.resize(_width, _plus);
      ModelElementContainer _container = container.getContainer();
      final ModelElementContainer it = _container;
      boolean _matched = false;
      if (it instanceof Container) {
        _matched=true;
        this.increaseIfNecessaryRight(((Container)it));
      }
    }
  }
  
  public void moveConnected(final Node node, final int deltaX, final int deltaY) {
    this.moveConnected_rec(node, deltaX, deltaY, CollectionLiterals.<Node>newHashSet(node));
  }
  
  public void moveConnected_rec(final Node node, final int deltaX, final int deltaY, final Set<Node> seen) {
    EList<Node> _successors = node.<Node>getSuccessors();
    EList<Node> _predecessors = node.<Node>getPredecessors();
    Iterable<Node> _plus = Iterables.<Node>concat(_successors, _predecessors);
    for (final Node n : _plus) {
      boolean _add = seen.add(n);
      if (_add) {
        int _x = n.getX();
        int _plus_1 = (_x + deltaX);
        int _y = n.getY();
        int _plus_2 = (_y + deltaY);
        n.move(_plus_1, _plus_2);
        this.moveConnected_rec(n, deltaX, deltaY, seen);
      }
    }
  }
  
  public void moveSuccessors(final Node node, final int deltaX, final int deltaY) {
    this.moveSuccessors_rec(node, deltaX, deltaY, CollectionLiterals.<Node>newHashSet(node));
  }
  
  public void moveSuccessors_rec(final Node node, final int deltaX, final int deltaY, final Set<Node> seen) {
    EList<Node> _successors = node.<Node>getSuccessors();
    for (final Node n : _successors) {
      boolean _add = seen.add(n);
      if (_add) {
        int _x = n.getX();
        int _plus = (_x + deltaX);
        int _y = n.getY();
        int _plus_1 = (_y + deltaY);
        n.move(_plus, _plus_1);
        this.moveSuccessors_rec(n, deltaX, deltaY, seen);
      }
    }
  }
  
  public void alignWithTopOf(final Node node, final Node otherNode) {
    node.move(node.getX(), otherNode.getY());
  }
  
  public void alignWithBottomOf(final Node node, final Node otherNode) {
    int _x = node.getX();
    int _y = otherNode.getY();
    int _height = otherNode.getHeight();
    int _plus = (_y + _height);
    int _height_1 = node.getHeight();
    int _minus = (_plus - _height_1);
    node.move(_x, _minus);
  }
  
  public void moveToBottomOf(final Node node, final Node otherNode, final int margin) {
    int _center = this.getCenter(otherNode);
    int _width = node.getWidth();
    int _divide = (_width / 2);
    int _minus = (_center - _divide);
    int _bottom = this.getBottom(otherNode);
    int _plus = (_bottom + margin);
    node.move(_minus, _plus);
  }
  
  public void moveToBottom(final Node it) {
    final int oldX = it.getX();
    final int oldY = it.getY();
    int _x = it.getX();
    int _bottom = this.getBottom(this.getBottomChild(it.getContainer()));
    int _plus = (_bottom + ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
    it.move(_x, _plus);
    int _x_1 = it.getX();
    final int deltaX = (_x_1 - oldX);
    int _y = it.getY();
    final int deltaY = (_y - oldY);
    if (((deltaX != 0) || (deltaY != 0))) {
      this.moveSuccessors(it, deltaX, deltaY);
    }
  }
  
  public void moveToRight(final Node it) {
    final int oldX = it.getX();
    final int oldY = it.getY();
    int _right = this.getRight(this.getBottomChild(it.getContainer()));
    int _plus = (_right + ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
    it.move(_plus, it.getY());
    int _x = it.getX();
    final int deltaX = (_x - oldX);
    int _y = it.getY();
    final int deltaY = (_y - oldY);
    if (((deltaX != 0) || (deltaY != 0))) {
      this.moveSuccessors(it, deltaX, deltaY);
    }
  }
  
  public void moveToBottomRight(final Node it) {
    final int oldX = it.getX();
    final int oldY = it.getY();
    int _right = this.getRight(this.getBottomChild(it.getContainer()));
    int _plus = (_right + ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
    int _bottom = this.getBottom(this.getBottomChild(it.getContainer()));
    int _plus_1 = (_bottom + ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
    it.move(_plus, _plus_1);
    int _x = it.getX();
    final int deltaX = (_x - oldX);
    int _y = it.getY();
    final int deltaY = (_y - oldY);
    if (((deltaX != 0) || (deltaY != 0))) {
      this.moveSuccessors(it, deltaX, deltaY);
    }
  }
  
  protected void _alignBeneathOf(final Node node, final Node otherNode) {
    this.moveToBottomOf(node, otherNode, ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
  }
  
  protected void _alignBeneathOf(final DataFlowTarget node, final DataFlowSource otherNode) {
    final int oldX = node.getX();
    final int oldY = node.getY();
    this.moveToBottomOf(node, otherNode, ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
    int _x = node.getX();
    final int deltaX = (_x - oldX);
    int _y = node.getY();
    final int deltaY = (_y - oldY);
    if (((deltaX != 0) || (deltaY != 0))) {
      boolean _matched = false;
      if (node instanceof SIB) {
        _matched=true;
        EList<Branch> _branchSuccessors = ((SIB)node).getBranchSuccessors();
        for (final Branch sibBranch : _branchSuccessors) {
          int _x_1 = sibBranch.getX();
          int _plus = (_x_1 + deltaX);
          int _y_1 = sibBranch.getY();
          int _plus_1 = (_y_1 + deltaY);
          sibBranch.move(_plus, _plus_1);
        }
      }
    }
  }
  
  protected void _alignBeneathOf(final DataFlowSource node, final DataFlowTarget otherNode) {
    this.moveToBottomOf(node, otherNode, ProcessModelGenerationLanguage.SIB_TO_BRANCH_OFFSET);
  }
  
  protected void _alignRightOf(final Node node, final Node otherNode) {
    this.moveToRightOf(node, otherNode, ProcessModelGenerationLanguage.SIB_TO_BRANCH_OFFSET);
  }
  
  protected void _alignRightOf(final DataFlowSource branch, final DataFlowTarget node) {
    final int oldX = branch.getX();
    final int oldY = branch.getY();
    this.moveToRightOf(branch, node, ProcessModelGenerationLanguage.SIB_TO_BRANCH_OFFSET);
    int _x = branch.getX();
    final int deltaX = (_x - oldX);
    int _y = branch.getY();
    final int deltaY = (_y - oldY);
    if (((deltaX != 0) || (deltaY != 0))) {
      boolean _matched = false;
      if (branch instanceof Branch) {
        _matched=true;
        EList<SIB> _sIBSuccessors = ((Branch)branch).getSIBSuccessors();
        for (final SIB sib : _sIBSuccessors) {
          int _x_1 = sib.getX();
          int _plus = (_x_1 + deltaX);
          int _y_1 = sib.getY();
          int _plus_1 = (_y_1 + deltaY);
          sib.move(_plus, _plus_1);
        }
      }
    }
  }
  
  protected void _alignRightOf(final DataFlowTarget node, final DataFlowSource branch) {
    this.moveToRightOf(node, branch, ProcessModelGenerationLanguage.BRANCH_TO_SIB_OFFSET);
  }
  
  public void moveToRightOf(final Node node, final Node otherNode, final int margin) {
    int _right = this.getRight(otherNode);
    int _plus = (_right + margin);
    int _middle = this.getMiddle(otherNode);
    int _height = node.getHeight();
    int _divide = (_height / 2);
    int _minus = (_middle - _divide);
    node.move(_plus, _minus);
  }
  
  public int getCenter(final Node it) {
    int _x = it.getX();
    int _width = it.getWidth();
    int _divide = (_width / 2);
    return (_x + _divide);
  }
  
  public int getBottom(final Node it) {
    int _y = it.getY();
    int _height = it.getHeight();
    return (_y + _height);
  }
  
  public int getRight(final Node it) {
    int _x = it.getX();
    int _width = it.getWidth();
    return (_x + _width);
  }
  
  public int getMiddle(final Node it) {
    int _y = it.getY();
    int _height = it.getHeight();
    int _divide = (_height / 2);
    return (_y + _divide);
  }
  
  public Node getBottomChild(final ModelElementContainer it) {
    final Function1<Node, Integer> _function = new Function1<Node, Integer>() {
      @Override
      public Integer apply(final Node it) {
        return Integer.valueOf(ProcessModelGenerationLanguage.this.getBottom(it));
      }
    };
    return IterableExtensions.<Node>head(ListExtensions.<Node>reverse(IterableExtensions.<Node, Integer>sortBy(it.getAllNodes(), _function)));
  }
  
  public Node getRightChild(final Container it) {
    final Function1<Node, Integer> _function = new Function1<Node, Integer>() {
      @Override
      public Integer apply(final Node it) {
        return Integer.valueOf(ProcessModelGenerationLanguage.this.getRight(it));
      }
    };
    return IterableExtensions.<Node>head(ListExtensions.<Node>reverse(IterableExtensions.<Node, Integer>sortBy(it.getAllNodes(), _function)));
  }
  
  public int beneathBottomAttribute(final Container con) {
    return this.beneathBottomChild(con, ProcessModelGenerationLanguage.ATTRIBUTE_MARGIN_Y, ProcessModelGenerationLanguage.VARIABLE_PADDING_TOP);
  }
  
  public int beneathBottomVariable(final Container con) {
    return this.beneathBottomChild(con, ProcessModelGenerationLanguage.VARIABLE_MARGIN_Y, ProcessModelGenerationLanguage.DATACONTEXT_PADDING_TOP);
  }
  
  public int beneathBottomChild(final Container con, final int margin, final int min) {
    int _xblockexpression = (int) 0;
    {
      final Node botChild = this.getBottomChild(con);
      int _xifexpression = (int) 0;
      if ((botChild != null)) {
        int _bottom = this.getBottom(botChild);
        _xifexpression = (_bottom + margin);
      } else {
        _xifexpression = min;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public int maxAttributeWidth(final Container con) {
    return this.maxChildWidth(con, ProcessModelGenerationLanguage.ATTRIBUTE_MARGIN_X);
  }
  
  public int maxVariableWidth(final Container con) {
    return this.maxChildWidth(con, ProcessModelGenerationLanguage.VARIABLE_MARGIN_X);
  }
  
  public int maxChildWidth(final Container con, final int padding) {
    int _width = con.getWidth();
    return (_width - (padding * 2));
  }
  
  public void applyLayout(final Edge edge, final EdgeLayout layout) {
    layout.apply(edge);
  }
  
  public EObject getByID(final String id) {
    return ReferenceRegistry.getInstance().getEObject(id);
  }
  
  public InputStatic setStaticInput(final InputPort inPort, final String value) {
    final String name = inPort.getName();
    final ModelElementContainer sib = inPort.getContainer();
    new PrimitivePortToStatic().execute(((PrimitiveInputPort) inPort));
    final Function1<InputStatic, Boolean> _function = new Function1<InputStatic, Boolean>() {
      @Override
      public Boolean apply(final InputStatic it) {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      }
    };
    final InputStatic staticInPort = this._graphModelExtension.<InputStatic>findThe(sib, InputStatic.class, _function);
    final InputStatic it = staticInPort;
    boolean _matched = false;
    if (it instanceof TextInputStatic) {
      _matched=true;
      ((TextInputStatic)it).setValue(value);
    }
    return staticInPort;
  }
  
  public void overtakeIncomingEdgesOf(final Node newNode, final Node node) {
    final Consumer<Edge> _function = new Consumer<Edge>() {
      @Override
      public void accept(final Edge it) {
        it.reconnectTarget(newNode);
      }
    };
    node.<Edge>getIncoming().forEach(_function);
  }
  
  public void overtakeOutgoingEdgesOf(final Node newNode, final Node node) {
    final Consumer<Edge> _function = new Consumer<Edge>() {
      @Override
      public void accept(final Edge it) {
        it.reconnectSource(newNode);
      }
    };
    node.<Edge>getOutgoing().forEach(_function);
  }
  
  public void overtakeBendpointsOf(final Edge newEdge, final Edge edge) {
    InternalModelElement _internalElement_ = edge.getInternalElement_();
    EList<_Point> _bendpoints = ((InternalEdge) _internalElement_).getBendpoints();
    for (final _Point bp : _bendpoints) {
      newEdge.addBendpoint(bp.getX(), bp.getY());
    }
  }
  
  public void applyManhattanLayout(final Edge it) {
    List<info.scce.dime.process.helper.EdgeLayoutUtils.Location> _manhattanPoints = info.scce.dime.process.helper.EdgeLayoutUtils.getManhattanPoints(it.getSourceElement(), it.getTargetElement());
    for (final info.scce.dime.process.helper.EdgeLayoutUtils.Location p : _manhattanPoints) {
      it.addBendpoint(p.x, p.y);
    }
  }
  
  public info.scce.dime.process.process.Attribute addAttribute(final ComplexVariable it, final Object attr) {
    if (attr instanceof info.scce.dime.data.data.ComplexAttribute) {
      return _addAttribute(it, (info.scce.dime.data.data.ComplexAttribute)attr);
    } else if (attr instanceof info.scce.dime.data.data.PrimitiveAttribute) {
      return _addAttribute(it, (info.scce.dime.data.data.PrimitiveAttribute)attr);
    } else if (attr instanceof String) {
      return _addAttribute(it, (String)attr);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it, attr).toString());
    }
  }
  
  public InputPort addInputPort(final Container dataFlowTarget, final Node attr) {
    if (attr instanceof info.scce.dime.data.data.ComplexAttribute) {
      return _addInputPort(dataFlowTarget, (info.scce.dime.data.data.ComplexAttribute)attr);
    } else if (attr instanceof info.scce.dime.data.data.PrimitiveAttribute) {
      return _addInputPort(dataFlowTarget, (info.scce.dime.data.data.PrimitiveAttribute)attr);
    } else if (attr instanceof ComplexVariable) {
      return _addInputPort(dataFlowTarget, (ComplexVariable)attr);
    } else if (attr instanceof PrimitiveVariable) {
      return _addInputPort(dataFlowTarget, (PrimitiveVariable)attr);
    } else if (attr instanceof Type) {
      return _addInputPort(dataFlowTarget, (Type)attr);
    } else if (attr instanceof ComplexAttribute) {
      return _addInputPort(dataFlowTarget, (ComplexAttribute)attr);
    } else if (attr instanceof PrimitiveAttribute) {
      return _addInputPort(dataFlowTarget, (PrimitiveAttribute)attr);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(dataFlowTarget, attr).toString());
    }
  }
  
  public OutputPort addOutputPort(final DataFlowSource dataFlowSource, final Node input) {
    if (input instanceof ComplexInputPort) {
      return _addOutputPort(dataFlowSource, (ComplexInputPort)input);
    } else if (input instanceof PrimitiveInputPort) {
      return _addOutputPort(dataFlowSource, (PrimitiveInputPort)input);
    } else if (input instanceof info.scce.dime.data.data.ComplexAttribute) {
      return _addOutputPort(dataFlowSource, (info.scce.dime.data.data.ComplexAttribute)input);
    } else if (input instanceof info.scce.dime.data.data.PrimitiveAttribute) {
      return _addOutputPort(dataFlowSource, (info.scce.dime.data.data.PrimitiveAttribute)input);
    } else if (input instanceof ComplexVariable) {
      return _addOutputPort(dataFlowSource, (ComplexVariable)input);
    } else if (input instanceof PrimitiveVariable) {
      return _addOutputPort(dataFlowSource, (PrimitiveVariable)input);
    } else if (input instanceof Type) {
      return _addOutputPort(dataFlowSource, (Type)input);
    } else if (input instanceof ComplexAttribute) {
      return _addOutputPort(dataFlowSource, (ComplexAttribute)input);
    } else if (input instanceof PrimitiveAttribute) {
      return _addOutputPort(dataFlowSource, (PrimitiveAttribute)input);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(dataFlowSource, input).toString());
    }
  }
  
  public DataFlow dataFlowTo(final Node outPort, final Node inPort) {
    if (outPort instanceof ComplexOutputPort
         && inPort instanceof ComplexInputPort) {
      return _dataFlowTo((ComplexOutputPort)outPort, (ComplexInputPort)inPort);
    } else if (outPort instanceof PrimitiveOutputPort
         && inPort instanceof PrimitiveInputPort) {
      return _dataFlowTo((PrimitiveOutputPort)outPort, (PrimitiveInputPort)inPort);
    } else if (outPort instanceof ComplexOutputPort
         && inPort instanceof ComplexVariable) {
      return _dataFlowTo((ComplexOutputPort)outPort, (ComplexVariable)inPort);
    } else if (outPort instanceof PrimitiveOutputPort
         && inPort instanceof PrimitiveVariable) {
      return _dataFlowTo((PrimitiveOutputPort)outPort, (PrimitiveVariable)inPort);
    } else if (outPort instanceof ComplexOutputPort
         && inPort instanceof ComplexAttribute) {
      return _dataFlowTo((ComplexOutputPort)outPort, (ComplexAttribute)inPort);
    } else if (outPort instanceof PrimitiveOutputPort
         && inPort instanceof PrimitiveAttribute) {
      return _dataFlowTo((PrimitiveOutputPort)outPort, (PrimitiveAttribute)inPort);
    } else if (outPort instanceof ComplexVariable
         && inPort instanceof ComplexInputPort) {
      return _dataFlowTo((ComplexVariable)outPort, (ComplexInputPort)inPort);
    } else if (outPort instanceof PrimitiveVariable
         && inPort instanceof PrimitiveInputPort) {
      return _dataFlowTo((PrimitiveVariable)outPort, (PrimitiveInputPort)inPort);
    } else if (outPort instanceof ComplexAttribute
         && inPort instanceof ComplexInputPort) {
      return _dataFlowTo((ComplexAttribute)outPort, (ComplexInputPort)inPort);
    } else if (outPort instanceof PrimitiveAttribute
         && inPort instanceof PrimitiveInputPort) {
      return _dataFlowTo((PrimitiveAttribute)outPort, (PrimitiveInputPort)inPort);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(outPort, inPort).toString());
    }
  }
  
  public DataFlow dataFlowFrom(final Node inPort, final Node outPort) {
    if (inPort instanceof ComplexInputPort
         && outPort instanceof ComplexOutputPort) {
      return _dataFlowFrom((ComplexInputPort)inPort, (ComplexOutputPort)outPort);
    } else if (inPort instanceof ComplexInputPort
         && outPort instanceof ComplexVariable) {
      return _dataFlowFrom((ComplexInputPort)inPort, (ComplexVariable)outPort);
    } else if (inPort instanceof PrimitiveInputPort
         && outPort instanceof PrimitiveVariable) {
      return _dataFlowFrom((PrimitiveInputPort)inPort, (PrimitiveVariable)outPort);
    } else if (inPort instanceof ComplexInputPort
         && outPort instanceof ComplexAttribute) {
      return _dataFlowFrom((ComplexInputPort)inPort, (ComplexAttribute)outPort);
    } else if (inPort instanceof PrimitiveInputPort
         && outPort instanceof PrimitiveAttribute) {
      return _dataFlowFrom((PrimitiveInputPort)inPort, (PrimitiveAttribute)outPort);
    } else if (inPort instanceof ComplexVariable
         && outPort instanceof ComplexOutputPort) {
      return _dataFlowFrom((ComplexVariable)inPort, (ComplexOutputPort)outPort);
    } else if (inPort instanceof PrimitiveVariable
         && outPort instanceof PrimitiveOutputPort) {
      return _dataFlowFrom((PrimitiveVariable)inPort, (PrimitiveOutputPort)outPort);
    } else if (inPort instanceof ComplexAttribute
         && outPort instanceof ComplexOutputPort) {
      return _dataFlowFrom((ComplexAttribute)inPort, (ComplexOutputPort)outPort);
    } else if (inPort instanceof PrimitiveAttribute
         && outPort instanceof PrimitiveOutputPort) {
      return _dataFlowFrom((PrimitiveAttribute)inPort, (PrimitiveOutputPort)outPort);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(inPort, outPort).toString());
    }
  }
  
  public void increaseIfNecessaryBottom(final Container container) {
    if (container instanceof DataContext) {
      _increaseIfNecessaryBottom((DataContext)container);
      return;
    } else if (container instanceof Variable) {
      _increaseIfNecessaryBottom((Variable)container);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(container).toString());
    }
  }
  
  public void increaseIfNecessaryRight(final Container container) {
    if (container instanceof DataContext) {
      _increaseIfNecessaryRight((DataContext)container);
      return;
    } else if (container instanceof Variable) {
      _increaseIfNecessaryRight((Variable)container);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(container).toString());
    }
  }
  
  public void alignBeneathOf(final Node node, final Node otherNode) {
    if (node instanceof DataFlowSource
         && otherNode instanceof DataFlowTarget) {
      _alignBeneathOf((DataFlowSource)node, (DataFlowTarget)otherNode);
      return;
    } else if (node instanceof DataFlowTarget
         && otherNode instanceof DataFlowSource) {
      _alignBeneathOf((DataFlowTarget)node, (DataFlowSource)otherNode);
      return;
    } else if (node != null
         && otherNode != null) {
      _alignBeneathOf(node, otherNode);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(node, otherNode).toString());
    }
  }
  
  public void alignRightOf(final Node branch, final Node node) {
    if (branch instanceof DataFlowSource
         && node instanceof DataFlowTarget) {
      _alignRightOf((DataFlowSource)branch, (DataFlowTarget)node);
      return;
    } else if (branch instanceof DataFlowTarget
         && node instanceof DataFlowSource) {
      _alignRightOf((DataFlowTarget)branch, (DataFlowSource)node);
      return;
    } else if (branch != null
         && node != null) {
      _alignRightOf(branch, node);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(branch, node).toString());
    }
  }
}
