/*-
 * #%L
 * DIME
 * %%
 * Copyright (C) 2021 - 2022 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package info.scce.dime.gui.actions

import info.scce.dime.api.DIMECustomAction
import info.scce.dime.gui.gui.GUI
import java.io.File
import org.eclipse.core.filesystem.EFS
import org.eclipse.core.filesystem.IFileStore
import org.eclipse.core.resources.IResource
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.ui.IEditorPart
import org.eclipse.ui.IEditorReference
import org.eclipse.ui.IWorkbenchPage
import org.eclipse.ui.IWorkbenchPartReference
import org.eclipse.ui.PartInitException
import org.eclipse.ui.PlatformUI
import org.eclipse.ui.ide.IDE

/** 
 * The open preview action opens a plain HTML5 preview for a given GUI model
 * if the HTML file is already generated by a resource change listener hook.
 * The context menu action can be used on the GUI model itself.
 * @author zweihoff
 */
class OpenPreview extends DIMECustomAction<GUI> {
	/** 
	 * Returns the name of the context menu entry
	 */
	override getName() {
		"Show preview"
	}

	/** 
	 * Preserves that the action does not perform any changes to the GUI model,
	 * so that it has not to be flagged as dirty
	 */
	override hasDoneChanges() {
		false
	}

	//TODO beautify after auto conversion to Xtend class

	/** 
	 * Opens the preview HTML file in the Eclipse browser.
	 */
	override void execute(GUI node) {
		val String guiName = node.getTitle()
		val String platformString = node.eResource().getURI().toPlatformString(true)
		val IResource ir = ResourcesPlugin::getWorkspace().getRoot().findMember(platformString)
		val File f = new File(ir.getProject().getLocation().append('''_tmp/«guiName».html'''.toString).toString())

		if (f.exists()) {
			var boolean found = false
			val IWorkbenchPage wp = PlatformUI::getWorkbench().getActiveWorkbenchWindow().getActivePage()
			for (IEditorReference er : PlatformUI::getWorkbench().getActiveWorkbenchWindow().getActivePage().
				getEditorReferences()) {
				if (er.getId().equals("org.eclipse.ui.browser.editor")) {
					var IEditorPart iep = er.getEditor(false)
					if (iep instanceof IWorkbenchPartReference) {
						var IWorkbenchPartReference iwpr = (iep as IWorkbenchPartReference)
						if (iwpr.getPartName().equals('''Preview «guiName»'''.toString)) {
							iwpr.getPart(true).setFocus()
						}
					}
				}
			}
			if (!found) async[
				var IFileStore file = EFS::getLocalFileSystem().getStore(f.toURI())
				try {
					IDE::openEditorOnFileStore(wp, file)
				} catch (PartInitException e) {
					e.printStackTrace()
				}
			]
		}
	}
}
