/*-
 * #%L
 * CINCO
 * %%
 * Copyright (C) 2021 TU Dortmund University - Department of Computer Science - Chair for Programming Systems
 * %%
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is
 * available at https://www.gnu.org/software/classpath/license.html.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 * #L%
 */
package de.jabc.cinco.meta.util.xapi

import java.net.URL
import java.util.function.Predicate
import org.eclipse.core.resources.IFile
import org.eclipse.core.runtime.IAdaptable
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.resource.Resource
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl
import org.eclipse.emf.edit.domain.IEditingDomainProvider
import org.eclipse.swt.widgets.Display
import org.eclipse.ui.IEditorInput
import org.eclipse.ui.IEditorPart
import org.eclipse.ui.IPathEditorInput
import org.eclipse.ui.IURIEditorInput
import org.eclipse.ui.PlatformUI
import org.eclipse.ui.part.FileEditorInput

import static org.eclipse.emf.common.util.URI.*

/**
 * Workbench-specific extension methods.
 * 
 * @author Steve Bosselmann
 */
class WorkbenchExtension {
	
	/**
	 * Retrieves the current workbench.
	 * 
	 * @return The current workbench. Might be {@code null}.
	 */
	def getWorkbench() {
		try {
			PlatformUI.workbench
		} catch(IllegalStateException e) {
			// do not print stacktrace nor console output
		}
	}
	
	/**
	 * Retrieves the active window of the workbench.
	 * 
	 * @return The active window, or {@code null} if it cannot be retrieved for
	 *   whatever reason.
	 */
	def  getActiveWorkbenchWindow() {
		if (Display.current === null)
			System.err.println("Trying to retrieve active workbench window without using Display thread")
		workbench?.activeWorkbenchWindow
	}

	/**
	 * Retrieves the active page of the active workbench window.
	 * Returns {@code null} if any of these entities cannot be retrieved instead
	 * of throwing exceptions.
	 * 
	 * @return The active page, or {@code null} if it cannot be retrieved for
	 *   whatever reason.
	 */
	def  getActivePage() {
		activeWorkbenchWindow?.activePage
	}
	
	/**
	 * Retrieves the active editor in the active page of the active workbench window.
	 * Returns {@code null} if any of these entities cannot be retrieved instead
	 * of throwing exceptions.
	 * 
	 * @return The active editor, or {@code null} if it cannot be retrieved for
	 *   whatever reason.
	 */
	def  getActiveEditor() {
		activePage?.activeEditor
	}
	
	/**
	 * Retrieves the first editor in the active page of the active workbench window,
	 * iff it fulfills the specified predicate.
	 * Returns {@code null} if any of these entities cannot be retrieved instead
	 * of throwing exceptions.
	 * 
	 * @return The first editor that fulfills the specified predicate, or {@code null}
	 *   if none is found or an editor cannot be retrieved for whatever reason.
	 */
	def  getEditor(Predicate<IEditorPart> predicate) {
		activePage?.editorReferences.findFirst[ref |
			predicate.test(ref.getEditor(true))
		]?.getEditor(true)
	}
	
	/**
	 * Retrieves the project that the resource currently edited in the specified
	 * editor is associated with.
	 * 
	 * @param editor for which to retrieve the associated project.
	 * @return The project corresponding to the editor, or {@code null} if not
	 *   existing.
	 */
	def getProject(IEditorPart editor) {
		val res = editor.resource
		if (res !== null) {
			extension val ResourceExtension = new ResourceExtension
			res.project
		}
		else null
	}
	
	/**
	 * Retrieves the resource currently edited in the specified editor.
	 * If the editor implements the interface {@code IEditingDomainProvider} the
	 * resource is retrieved directly. If not or if the editing domain is
	 * {@code null}, the resource is retrieved via the file that is the input of
	 * the specified editor.
	 * Returns {@code null} if any of these entities cannot be retrieved instead
	 * of throwing exceptions.
	 * 
	 * @param editor for which to retrieve the associated resource.
	 * @return The resource corresponding to the editor, or {@code null} if not
	 *   existing.
	 */
	def Resource getResource(IEditorPart editor) {
		val domain = editor.editingDomain
		if (domain !== null) {
			domain.resourceSet.resources.head
		} else {
			extension val FileExtension = new FileExtension
			editor.file?.resource
		}
	}
	
	/**
	 * Retrieves the editing domain of the specified editor.
	 * 
	 * @param editor for which to retrieve the associated editing domain.
	 * @return The editing domain of the specified editor, or {@code null} if
	 * not existing or the editor does not implement the interface
	 * {@link IEditingDomainProvider}.
	 */
	def getEditingDomain(IEditorPart editor) {
		switch editor {
			IEditingDomainProvider: editor.editingDomain
		}
	}
	
	/**
	 * Retrieves the file that represents the input of the specified editor.
	 * 
	 * @param editor for which to retrieve the associated input file.
	 * @return The file that represents the input of the specified editor or
	 * {@code null} if not existing or the editor does not implement the interface
	 * {@link IPathEditorInput}.
	 */
	def getFile(IEditorPart editor) {
		switch input : editor?.editorInput {
			FileEditorInput: input.file
			IPathEditorInput: {
				extension val WorkspaceExtension = new WorkspaceExtension
				val files = workspaceRoot.getFiles[
					fullPath == input.path || location == input.path
				]
				if (files.size == 1)
					return files.get(0)
			}
		}
	}
	
	/**
	 * Retrieves the URI of the input of an editor. This can be both, the URI of a
	 * file in the workspace (input is {@link IURIEditorInput}) as well an abstract
	 * URI pointing to a resource in an archive or anywhere else (input is
	 * {@link IAdaptable}).
	 * 
	 * @param input of the editor.
	 * @return The URI of the input.
	 */
	def URI getURI(IEditorInput input) {
		switch input {
			IURIEditorInput: createURI(new URL(input.URI.toString).toString)
			IAdaptable: {
				val file = input.getAdapter(IFile)
				if (file !== null) {
					new ResourceSetImpl().getURIConverter().normalize(
						createPlatformResourceURI(file.fullPath.toString, true)
					)
				}
			}
		}
	}
	
	/**
	 * Retrieve the current instance of Display, the default instance otherwise.
	 * Display is responsible for managing the connection between SWT and the
	 * underlying operating system.
	 */
	def getDisplay() {
		Display.current ?: Display.^default
	}
	
	/**
	 * Causes the run() method of the runnable to be invoked by the UI thread
	 * at the next reasonable opportunity. The caller of this method continues
	 * to run in parallel.
	 */
	def async(Runnable runnable) {
		display.asyncExec(runnable)
	}
	
	/**
	 * Causes the run() method of the runnable to be invoked by the UI thread
	 * at the next reasonable opportunity. The thread which calls this method
	 * is suspended until the runnable completes.
	 */
	def sync(Runnable runnable) {
		display.syncExec(runnable)
	}
}
